# -*- coding: utf-8 -*-
import os
import re
import html
import uuid
import sqlite3
import logging
from datetime import datetime
import threading

import requests
from requests_pkcs12 import Pkcs12Adapter
from lxml import etree
from signxml import XMLSigner, methods
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.serialization import pkcs12, Encoding
from OpenSSL.crypto import load_certificate, FILETYPE_PEM
from nfse_db import upsert_status_nf, log_evento

# ============ CONFIG ============
CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

HOMOLOG_URL = "https://bhisshomologaws.pbh.gov.br/bhiss-ws/nfse"
PROD_URL    = "https://bhissws.pbh.gov.br/bhiss-ws/nfse"

# ABRASF 1.00 unificado
NS_DADOS = "http://www.abrasf.org.br/nfse.xsd"
NS_DS    = "http://www.w3.org/2000/09/xmldsig#"
etree.register_namespace("ds", NS_DS)

BASE_DIR = os.path.dirname(os.path.abspath(__file__))
XSD_PATH = os.path.join(BASE_DIR, "schemas", "pbh", "nfse.xsd")  # opcional (se existir)

logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(levelname)s - %(message)s")

# ============ Helpers ============
PBH_DICAS = {
    "E174": "Erro na assinatura. Assine InfPedidoCancelamento e coloque ds:Signature como irmã dentro de <Pedido>. "
            "Transforms: enveloped, depois exclusive c14n; Reference URI apontando para o Id correto.",
    "E181": "Namespace da assinatura incorreto. Use 'http://www.w3.org/2000/09/xmldsig#' em toda a árvore.",
}

_SIG_LOCALS = {
    "Signature", "SignedInfo", "CanonicalizationMethod", "SignatureMethod",
    "Reference", "Transforms", "Transform", "DigestMethod", "DigestValue",
    "SignatureValue", "KeyInfo", "X509Data", "X509Certificate"
}
_DEF_ENV  = "http://www.w3.org/2000/09/xmldsig#enveloped-signature"
_DEF_C14N = "http://www.w3.org/2001/10/xml-exc-c14n#"

def _competencias_para_nf(numero_nfse: str):
    comp_serv, comp_nfse = None, None
    with sqlite3.connect(CAMINHO_BANCO) as conn:
        cur = conn.cursor()
        # Na nfse_notas, 'competencia' é a do serviço (seu script já grava assim)
        cur.execute("""
            SELECT competencia, data_emissao_nfse
            FROM nfse_notas
            WHERE numero_nfse = ?
            ORDER BY id DESC LIMIT 1
        """, (str(int(numero_nfse)),))
        row = cur.fetchone()
        if row:
            comp_serv = (row[0] or "").strip() or None
            de = (row[1] or "").strip()
            # tenta montar MM/YYYY a partir da data de emissão retornada pela PBH
            try:
                # Pega só a parte 'YYYY-MM-DD...' se vier com TZ
                comp_nfse = datetime.fromisoformat(de[:19]).strftime("%m/%Y")
            except Exception:
                try:
                    comp_nfse = datetime.strptime(de[:10], "%Y-%m-%d").strftime("%m/%Y")
                except Exception:
                    comp_nfse = None
    return comp_serv, comp_nfse

def _registrar_status(convenio_nome,
                      competencia=None,                  # legado
                      status=None,
                      numero_nfse=None,
                      caminho_xml="",
                      mensagem="",
                      *,
                      competencia_servico=None,          # novo
                      competencia_nfse=None):            # novo
    """
    Grava/atualiza o status da NF:
      - Usa competencia_servico como referência (ex.: 06/2025).
      - Preenche também a coluna antiga 'competencia' se ela existir (evita NOT NULL/UNIQUE).
      - Se existir 'competencia_nfse', grava também (ex.: 08/2025).
    """
    ts = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    msg = (mensagem or "")[:3000]
    num = str(numero_nfse or "").strip()

    # compat: se só veio 'competencia' (legado), use como 'competencia_servico'
    if not competencia_servico:
        competencia_servico = competencia
    if not competencia_servico:
        competencia_servico = datetime.now().strftime("%m/%Y")

    with sqlite3.connect(CAMINHO_BANCO) as conn:
        cur = conn.cursor()
        cur.execute("PRAGMA table_info(notas_emitidas_status)")
        cols = {row[1] for row in cur.fetchall()}

        tem_comp_serv = "competencia_servico" in cols
        tem_comp_lega = "competencia" in cols
        tem_comp_nfse = "competencia_nfse" in cols

        # --------- UPDATE (tenta casar por qualquer uma das competências) ---------
        set_parts = [
            "status = ?",
            "data_emissao = ?",
            "caminho_xml = COALESCE(?, caminho_xml)",
            "mensagem_erro = ?",
            "numero_nfse = ?",
        ]
        params_update = [status, ts, (caminho_xml or None), msg, num]

        if tem_comp_nfse:
            set_parts.append("competencia_nfse = COALESCE(?, competencia_nfse)")
            params_update.append(competencia_nfse)

        # também mantém a coluna antiga 'competencia' alinhada, se existir
        if tem_comp_lega:
            set_parts.append("competencia = ?")
            params_update.append(competencia_servico)

        where_parts = ["convenio = ?"]
        where_params = [convenio_nome]

        if tem_comp_serv and tem_comp_lega:
            where_parts.append("(competencia_servico = ? OR competencia = ?)")
            where_params.extend([competencia_servico, competencia_servico])
        elif tem_comp_serv:
            where_parts.append("competencia_servico = ?")
            where_params.append(competencia_servico)
        elif tem_comp_lega:
            where_parts.append("competencia = ?")
            where_params.append(competencia_servico)
        else:
            raise RuntimeError("Tabela notas_emitidas_status não possui coluna de competência.")

        sql_upd = f"""
            UPDATE notas_emitidas_status
               SET {", ".join(set_parts)}
             WHERE {" AND ".join(where_parts)}
        """
        cur.execute(sql_upd, params_update + where_params)
        rows = cur.rowcount

        # --------- INSERT se não atualizou nada ---------
        if rows == 0:
            insert_cols = ["convenio", "status", "data_emissao", "caminho_xml", "mensagem_erro", "numero_nfse"]
            insert_vals = [convenio_nome, status, ts, (caminho_xml or None), msg, num]

            # sempre que existirem, preencha as duas competências
            if tem_comp_serv:
                insert_cols.append("competencia_servico")
                insert_vals.append(competencia_servico)
            if tem_comp_lega:
                insert_cols.append("competencia")
                insert_vals.append(competencia_servico)
            if tem_comp_nfse:
                insert_cols.append("competencia_nfse")
                insert_vals.append(competencia_nfse)

            placeholders = ",".join(["?"] * len(insert_cols))
            sql_ins = f"INSERT INTO notas_emitidas_status ({', '.join(insert_cols)}) VALUES ({placeholders})"
            cur.execute(sql_ins, insert_vals)

        conn.commit()

        
def _atualizar_flags_nf(numero_nfse: str, *, cancelada: int | None = None,
                        cancelamento_pendente: int | None = None):
    """Atualiza colunas de estado em nfse_notas apenas se existirem."""
    with sqlite3.connect(CAMINHO_BANCO) as conn:
        cur = conn.cursor()
        cur.execute("PRAGMA table_info(nfse_notas)")
        cols = {r[1] for r in cur.fetchall()}

        sets, params = [], []
        if "cancelada" in cols and cancelada is not None:
            sets.append("cancelada = ?")
            params.append(int(cancelada))
            if cancelada == 1 and "data_cancelamento" in cols:
                sets.append("data_cancelamento = ?")
                params.append(datetime.now().strftime("%Y-%m-%d %H:%M:%S"))
        if "cancelamento_pendente" in cols and cancelamento_pendente is not None:
            sets.append("cancelamento_pendente = ?")
            params.append(int(cancelamento_pendente))

        if sets:
            params.append(str(int(numero_nfse)))
            sql = f"UPDATE nfse_notas SET {', '.join(sets)} WHERE numero_nfse = ?"
            cur.execute(sql, params)
            conn.commit()



def only_digits(s): 
    return re.sub(r"\D", "", str(s or ""))

def montar_cabecalho_nfse(versao="1.00") -> str:
    cab = etree.Element(f"{{{NS_DADOS}}}cabecalho", versao=versao, nsmap={None: NS_DADOS})
    etree.SubElement(cab, f"{{{NS_DADOS}}}versaoDados").text = versao
    return etree.tostring(cab, encoding="utf-8", method="xml").decode("utf-8")

def forcar_namespace_xmlsig(root: etree._Element):
    for el in root.iter():
        try:
            qn = etree.QName(el)
            local, uri = qn.localname, qn.namespace
        except Exception:
            continue
        if local in _SIG_LOCALS and uri != NS_DS:
            el.tag = f"{{{NS_DS}}}{local}"
    return root

def _fix_signature(sig_el: etree._Element) -> None:
    si = sig_el.find(f".//{{{NS_DS}}}SignedInfo")
    if si is None:
        return
    cm = si.find(f".//{{{NS_DS}}}CanonicalizationMethod")
    if cm is not None:
        cm.set("Algorithm", _DEF_C14N)
    ref = si.find(f".//{{{NS_DS}}}Reference")
    if ref is None:
        return
    tr_parent = ref.find(f".//{{{NS_DS}}}Transforms")
    if tr_parent is None:
        return
    for t in list(tr_parent):
        tr_parent.remove(t)
    etree.SubElement(tr_parent, f"{{{NS_DS}}}Transform").set("Algorithm", _DEF_ENV)
    etree.SubElement(tr_parent, f"{{{NS_DS}}}Transform").set("Algorithm", _DEF_C14N)

def extrair_mensagens_pbh(inner_xml: str):
    try:
        root = etree.fromstring(inner_xml.encode("utf-8"))
    except Exception:
        return []
    ns = {"n": NS_DADOS}
    msgs = []
    for m in root.findall(".//n:ListaMensagemRetorno/n:MensagemRetorno", namespaces=ns):
        cod = (m.findtext("n:Codigo", default="", namespaces=ns) or "").strip()
        msg = re.sub(r"\s+", " ", (m.findtext("n:Mensagem", default="", namespaces=ns) or "")).strip()
        cor = re.sub(r"\s+", " ", (m.findtext("n:Correcao", default="", namespaces=ns) or "")).strip()
        dicas = PBH_DICAS.get(cod, "")
        msgs.append({"codigo": cod, "mensagem": msg, "correcao": cor, "dica": dicas})
    return msgs

def raise_if_pbh_error(inner_xml: str):
    msgs = extrair_mensagens_pbh(inner_xml)
    if msgs:
        m = msgs[0]
        raise Exception(
            f"PBH [{m['codigo']}]: {m['mensagem']}"
            + (f" | Correção: {m['correcao']}" if m['correcao'] else "")
            + (f" | Dica: {m['dica']}" if m['dica'] else "")
        )

def validar_estrutura_xml(xml_path: str, xsd_path: str, log_path: str) -> bool:
    try:
        schema = etree.XMLSchema(etree.parse(xsd_path))
        schema.assertValid(etree.parse(xml_path))
        with open(log_path, "w", encoding="utf-8") as f:
            f.write("✅ Validação OK: XML é válido conforme o XSD\n")
        return True
    except Exception as e:
        with open(log_path, "w", encoding="utf-8") as f:
            f.write(f"❌ XML inválido conforme XSD.\n• {str(e)}\n")
        return False

def confirmacao_ok(inner_xml: str) -> bool:
    """Confirma se o retorno contém efetivamente uma confirmação de cancelamento."""
    return ("<RetCancelamento" in inner_xml and
            "<Confirmacao" in inner_xml and
            "<ListaMensagemRetorno" not in inner_xml)

# ============ Geração + Assinatura ============
def gerar_xml_cancelamento(*, numero_nfse: str, cnpj_prest: str, im_prest: str,
                           codigo_municipio: str = "3106200",
                           codigo_cancelamento: str = "2",
                           caminho_saida: str = "notas_emitidas/cancelamentos/cancelamento.xml") -> str:
    """
    Monta CancelarNfseEnvio (ABRASF 1.00) com namespace único e Id em InfPedidoCancelamento.
    """
    os.makedirs(os.path.dirname(caminho_saida) or ".", exist_ok=True)

    root = etree.Element(f"{{{NS_DADOS}}}CancelarNfseEnvio", nsmap={None: NS_DADOS})
    pedido = etree.SubElement(root, f"{{{NS_DADOS}}}Pedido")
    id_inf = f"canc{uuid.uuid4().hex[:8]}"
    inf = etree.SubElement(pedido, f"{{{NS_DADOS}}}InfPedidoCancelamento", Id=id_inf)

    ident = etree.SubElement(inf, f"{{{NS_DADOS}}}IdentificacaoNfse")
    etree.SubElement(ident, f"{{{NS_DADOS}}}Numero").text = str(int(numero_nfse))
    etree.SubElement(ident, f"{{{NS_DADOS}}}Cnpj").text = only_digits(cnpj_prest)   # <Cnpj> direto
    etree.SubElement(ident, f"{{{NS_DADOS}}}InscricaoMunicipal").text = only_digits(im_prest)
    etree.SubElement(ident, f"{{{NS_DADOS}}}CodigoMunicipio").text = only_digits(codigo_municipio)

    # Segurança: valide estrutura
    if ident.find(f"{{{NS_DADOS}}}Cnpj") is None or ident.find(f"{{{NS_DADOS}}}CpfCnpj") is not None:
        raise ValueError("Estrutura de IdentificacaoNfse inválida (Cnpj esperado, sem CpfCnpj).")

    etree.SubElement(inf, f"{{{NS_DADOS}}}CodigoCancelamento").text = only_digits(codigo_cancelamento)

    etree.ElementTree(root).write(caminho_saida, encoding="utf-8", xml_declaration=True, pretty_print=False)
    return caminho_saida

def assinar_pedido_cancelamento(xml_path: str, cert_path: str, senha_cert: str) -> str:
    """
    Assina <InfPedidoCancelamento Id="..."> (RSA-SHA1/SHA1, exclusive c14n)
    e injeta <ds:Signature> como IRMÃ de <InfPedidoCancelamento> dentro de <Pedido>.
    """
    with open(xml_path, "rb") as f:
        root = etree.parse(f).getroot()

    ns = {"n": NS_DADOS, "ds": NS_DS}
    pedido = root.find(".//n:Pedido", namespaces=ns)
    inf = root.find(".//n:Pedido/n:InfPedidoCancelamento", namespaces=ns)
    if pedido is None or inf is None:
        raise Exception("Estrutura inválida: esperado Pedido/InfPedidoCancelamento.")

    inf_id = inf.get("Id")
    if not inf_id:
        raise Exception("InfPedidoCancelamento precisa de atributo Id.")

    # carrega chave/cert
    with open(cert_path, "rb") as f:
        pfx = f.read()
    private_key, certificate, _ = pkcs12.load_key_and_certificates(
        pfx, senha_cert.encode(), backend=default_backend()
    )
    if not private_key or not certificate:
        raise Exception("Falha ao carregar chave/cert do PFX.")
    pem = certificate.public_bytes(Encoding.PEM)
    openssl_cert = load_certificate(FILETYPE_PEM, pem)

    signer = XMLSigner(
        method=methods.enveloped,
        signature_algorithm="rsa-sha1",
        digest_algorithm="sha1",
        c14n_algorithm=_DEF_C14N,
    )

    # Assina uma CÓPIA do Inf, coleta somente a <Signature>
    inf_copy = etree.fromstring(etree.tostring(inf))
    signed = signer.sign(inf_copy, key=private_key, cert=[openssl_cert], reference_uri=f"#{inf_id}")

    # Localiza a Signature
    if isinstance(signed.tag, str) and etree.QName(signed).localname == "Signature" and etree.QName(signed).namespace == NS_DS:
        sig = signed
    else:
        sig = signed.find(f".//{{{NS_DS}}}Signature")
    if sig is None:
        raise Exception("Falha ao gerar assinatura do pedido de cancelamento.")

    # Insere a assinatura como irmã de InfPedidoCancelamento
    pedido.insert(pedido.index(inf) + 1, sig)

    # Normaliza namespace ds + corrige ordem de transforms e c14n
    forcar_namespace_xmlsig(root)
    for s in root.xpath("//ds:Signature", namespaces=ns):
        _fix_signature(s)

    etree.ElementTree(root).write(xml_path, encoding="utf-8", xml_declaration=True, pretty_print=False)
    return xml_path

# ============ Envio SOAP ============
def enviar_cancelamento(xml_cancel_assinado_path: str, cert_path: str, senha_cert: str, url: str = HOMOLOG_URL) -> str:
    with open(xml_cancel_assinado_path, "r", encoding="utf-8") as f:
        raw = f.read()
    dados_xml = re.sub(r"<\?xml.*?\?>", "", raw).lstrip()
    cabecalho_xml = montar_cabecalho_nfse("1.00")

    envelope = f"""<?xml version="1.0" encoding="UTF-8"?>
<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/"
                  xmlns:ws="http://ws.bhiss.pbh.gov.br">
  <soapenv:Header/>
  <soapenv:Body>
    <ws:CancelarNfseRequest>
      <nfseCabecMsg><![CDATA[{cabecalho_xml}]]></nfseCabecMsg>
      <nfseDadosMsg><![CDATA[{dados_xml}]]></nfseDadosMsg>
    </ws:CancelarNfseRequest>
  </soapenv:Body>
</soapenv:Envelope>"""

    base = os.path.splitext(xml_cancel_assinado_path)[0]
    req_path  = base + "_soap_request.xml"
    resp_path = base + "_soap_response.xml"
    out_path  = base + "_outputXML.xml"

    with open(req_path, "w", encoding="utf-8") as f:
        f.write(envelope)

    s = requests.Session()
    s.mount("https://", Pkcs12Adapter(pkcs12_filename=cert_path, pkcs12_password=senha_cert))
    r = s.post(
        url,
        data=envelope.encode("utf-8"),
        headers={
            "Content-Type": "text/xml; charset=UTF-8",
            "SOAPAction": "http://ws.bhiss.pbh.gov.br/CancelarNfse",
        },
        verify=True,
        timeout=(15, 90),
    )

    with open(resp_path, "wb") as f:
        f.write(r.content)
    if r.status_code != 200:
        raise Exception(f"Erro HTTP {r.status_code}: {r.text[:800]}")

    soap = etree.fromstring(r.content)
    fault = soap.find(".//{http://schemas.xmlsoap.org/soap/envelope/}Fault")
    if fault is not None:
        fs = (fault.findtext("faultstring") or "").strip()
        fd = (fault.findtext("detail") or "").strip()
        raise Exception(f"SOAP Fault: {fs or 'Falha desconhecida'} | {fd}")

    out = soap.find(".//{*}outputXML")
    if out is None or not (out.text or "").strip():
        raise Exception("Sem <outputXML> na resposta do CancelarNfse.")
    inner = html.unescape(out.text or "")

    with open(out_path, "w", encoding="utf-8") as f:
        f.write(inner)

    # Erros funcionais PBH
    raise_if_pbh_error(inner)

    return inner  # XML interno como string

# ============ Orquestração ============
CANCEL_MOTIVOS = {
    "2": "Serviço não concluído"
}

def cancelar_nfse_por_convenio(
    numero_nfse: str,
    id_convenio: int,
    codigo_cancelamento: str = "2",
    url_envio: str = HOMOLOG_URL,
    competencia_servico: str | None = None
) -> tuple[bool, str]:
    """
    Cancela uma NFS-e no padrão PBH ABRASF 1.00.
    Alinhado ao fluxo do emissor: síncrono, salva arquivos, registra status imediatamente
    e atualiza flags em nfse_notas.
    """
    if not str(numero_nfse).strip().isdigit():
        return False, "Número da NFS-e inválido."
    if codigo_cancelamento not in CANCEL_MOTIVOS:
        return False, "Código de cancelamento inválido (use 1..5)."

    try:
        # 1) Dados do prestador e convênio
        with sqlite3.connect(CAMINHO_BANCO) as conn:
            cur = conn.cursor()

            cur.execute("""
                SELECT certificado_path, senha_certificado, cnpj, inscricao_municipal, codigo_municipio_ibge
                FROM medical_laudos
                LIMIT 1
            """)
            row = cur.fetchone()
            if not row:
                return False, "Configuração do prestador não encontrada."
            certificado_path, senha_certificado, cnpj_prest, im_prest, cod_mun_prest = row
            cod_mun_prest = re.sub(r"\D", "", (cod_mun_prest or ""))[:7] or "3106200"

            cur.execute("SELECT nome FROM convenios WHERE id = ?", (id_convenio,))
            r = cur.fetchone()
            if not r:
                return False, "Convênio não encontrado."
            convenio_nome = r[0]

        # 2) Descobrir competências reais da NF
        comp_serv, comp_nfse = _competencias_para_nf(numero_nfse)

        # fallback da competência de serviço
        if not comp_serv:
            if competencia_servico:
                comp_serv = competencia_servico
            else:
                with sqlite3.connect(CAMINHO_BANCO) as conn:
                    cur = conn.cursor()
                    cur.execute("""
                        SELECT competencia
                          FROM nfse_lotes
                         WHERE convenio_id = ?
                      ORDER BY id DESC LIMIT 1
                    """, (id_convenio,))
                    r = cur.fetchone()
                    comp_serv = (r[0] if r else None) or datetime.now().strftime("%m/%Y")

        # 3) Montar e assinar XML — salvar dentro da pasta do convênio/competência
        safe_conv = re.sub(r"[^0-9A-Za-z_-]+", "_", (convenio_nome or "").strip())
        safe_comp = (comp_serv or datetime.now().strftime("%m/%Y")).replace("/", "-")

        # Pasta: notas_emitidas/<Convênio>_<Competência>/cancelamento/
        base_dir = os.path.join("notas_emitidas", f"{safe_conv}_{safe_comp}", "cancelamento")
        os.makedirs(base_dir, exist_ok=True)

        # Nome-base: <Convênio>_<Competência>_cancelamento_nfse_<número>
        base_name = f"{safe_conv}_{safe_comp}_cancelamento_nfse_{int(numero_nfse)}"
        xml_cancel = os.path.join(base_dir, f"{base_name}.xml")



        gerar_xml_cancelamento(
            numero_nfse=str(int(numero_nfse)),
            cnpj_prest=cnpj_prest,
            im_prest=im_prest,
            codigo_municipio=cod_mun_prest,
            codigo_cancelamento=codigo_cancelamento,
            caminho_saida=xml_cancel
        )

        if os.path.exists(XSD_PATH):
            log_path_pre = xml_cancel.replace(".xml", "_pre_validacao.log")
            validar_estrutura_xml(xml_cancel, XSD_PATH, log_path_pre)

        assinar_pedido_cancelamento(xml_cancel, certificado_path, senha_certificado)

        # 4) Pré-status + flag pendente + log
        _registrar_status(
            convenio_nome,
            status="Cancelamento solicitado",
            numero_nfse=numero_nfse,
            caminho_xml=xml_cancel,
            competencia_servico=comp_serv,
            competencia_nfse=comp_nfse
        )
        _atualizar_flags_nf(numero_nfse, cancelamento_pendente=1)

        with sqlite3.connect(CAMINHO_BANCO) as conn:
            try:
                log_evento(conn, str(numero_nfse), "cancelamento_solicitado",
                           mensagem=f"Cancelamento solicitado para NF {numero_nfse}.",
                           payload_path=xml_cancel)
            except Exception:
                pass

        # 5) Enviar (síncrono) e classificar retorno
        inner = enviar_cancelamento(xml_cancel, certificado_path, senha_certificado, url=url_envio)

        # caminhos base (iguais ao emissor)
        base_prefix = os.path.splitext(xml_cancel)[0]
        out_path = base_prefix + "_outputXML.xml"

        with sqlite3.connect(CAMINHO_BANCO) as conn:
            try:
                log_evento(conn, str(numero_nfse), "cancelar_nfse_enviado",
                           mensagem="Requisição enviada ao webservice.",
                           payload_path=base_prefix + "_soap_request.xml")
                log_evento(conn, str(numero_nfse), "cancelar_nfse_retorno",
                           mensagem="Retorno recebido do webservice.",
                           payload_path=out_path)
            except Exception:
                pass

        # Se não tínhamos comp_nfse, tenta novamente após retorno
        if not comp_nfse:
            _, comp_nfse_local = _competencias_para_nf(numero_nfse)
        else:
            comp_nfse_local = comp_nfse

        if confirmacao_ok(inner):
            # Cancelamento efetivado
            _registrar_status(
                convenio_nome,
                status="Cancelada",
                numero_nfse=numero_nfse,
                caminho_xml=xml_cancel,
                mensagem=inner,
                competencia_servico=comp_serv,
                competencia_nfse=comp_nfse_local
            )
            _atualizar_flags_nf(numero_nfse, cancelada=1, cancelamento_pendente=0)

            # espelha no painel resumido
            try:
                upsert_status_nf(
                    convenio=convenio_nome,
                    competencia_servico=comp_serv,
                    status="Cancelada",
                    numero_nfse=str(int(numero_nfse)),
                    competencia_nfse=comp_nfse_local
                )
            except Exception:
                pass

            return True, f"NFS-e {int(numero_nfse)} cancelada com sucesso."
        else:
            # Em análise pela PBH
            _registrar_status(
                convenio_nome,
                status="Cancelamento em análise",
                numero_nfse=numero_nfse,
                caminho_xml=xml_cancel,
                mensagem=inner,
                competencia_servico=comp_serv,
                competencia_nfse=comp_nfse_local
            )
            # mantém pendente=1
            try:
                upsert_status_nf(
                    convenio=convenio_nome,
                    competencia_servico=comp_serv,
                    status="Cancelamento em análise",
                    numero_nfse=str(int(numero_nfse)),
                    competencia_nfse=comp_nfse_local
                )
            except Exception:
                pass

            return True, f"Cancelamento da NFS-e {int(numero_nfse)} registrado e em análise pela PBH."

    except Exception as e:
        logging.exception("Falha no cancelamento")
        # Reflete erro em status e limpa pendência
        try:
            _registrar_status(
                convenio_nome if 'convenio_nome' in locals() else "",
                status="Erro no cancelamento",
                numero_nfse=numero_nfse,
                caminho_xml=(xml_cancel if 'xml_cancel' in locals() else ""),
                mensagem=str(e),
                competencia_servico=(comp_serv if 'comp_serv' in locals() else None),
                competencia_nfse=(comp_nfse if 'comp_nfse' in locals() else None)
            )
            _atualizar_flags_nf(numero_nfse, cancelamento_pendente=0)
            try:
                upsert_status_nf(
                    convenio=(convenio_nome if 'convenio_nome' in locals() else ""),
                    competencia_servico=(comp_serv if 'comp_serv' in locals() else datetime.now().strftime("%m/%Y")),
                    status="Erro no cancelamento",
                    numero_nfse=str(int(numero_nfse)) if str(numero_nfse).isdigit() else None,
                    competencia_nfse=(comp_nfse if 'comp_nfse' in locals() else None)
                )
            except Exception:
                pass
        except Exception:
            pass
        return False, f"Falha ao cancelar NFS-e {numero_nfse}: {e}"



# ============ CLI de teste ============
if __name__ == "__main__":
    ok, msg = cancelar_nfse_por_convenio(
        numero_nfse="45",
        id_convenio=4,
        codigo_cancelamento="2",
        url_envio=HOMOLOG_URL
    )
    print(ok, msg)
