# db/inicializa_banco.py

import sqlite3
import os

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")


# ----------------------------
# Helpers
# ----------------------------
def get_table_columns(conn, table_name: str):
    cur = conn.cursor()
    cur.execute(f"PRAGMA table_info({table_name})")
    return [r[1] for r in cur.fetchall()]

def add_column_if_not_exists(conn, table: str, column_name: str, column_sql: str):
    try:
        cols = get_table_columns(conn, table)
    except sqlite3.OperationalError:
        return  # tabela ainda não existe; quem cria é o CREATE TABLE
    if column_name not in cols:
        cur = conn.cursor()
        cur.execute(f"ALTER TABLE {table} ADD COLUMN {column_sql}")
        conn.commit()
        print(f"Coluna '{column_name}' adicionada em {table}.")
    else:
        print(f"Coluna '{column_name}' já existe em {table}.")


# ----------------------------
# Impostos: migração robusta
# ----------------------------
def ensure_impostos_schema(conn):
    """
    Garante que 'cadastro_impostos' esteja no formato:
      id, nome, origem, valor, dia_vencimento, ativo
    Migra de qualquer formato antigo sem referenciar colunas inexistentes em SQL.
    """
    cur = conn.cursor()

    # Tabela existe?
    cur.execute("SELECT name FROM sqlite_master WHERE type='table' AND name='cadastro_impostos'")
    exists = cur.fetchone() is not None

    if not exists:
        # Cria do zero no formato correto
        cur.execute("""
            CREATE TABLE IF NOT EXISTS cadastro_impostos (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                nome TEXT NOT NULL,
                origem TEXT CHECK(origem IN ('Nacional','Estadual','Municipal')) NOT NULL,
                valor REAL NOT NULL,
                dia_vencimento INTEGER NOT NULL,
                ativo INTEGER DEFAULT 1
            )
        """)
        conn.commit()
        print("[impostos] Tabela 'cadastro_impostos' criada (formato correto).")
        return

    # Se existe, checa as colunas
    cols = set(get_table_columns(conn, "cadastro_impostos"))
    expected = {"id", "nome", "origem", "valor", "dia_vencimento", "ativo"}

    if expected.issubset(cols) and len(cols) >= 6:
        # Já está ok
        return

    print(f"[impostos] Formato atual de 'cadastro_impostos': {sorted(cols)}")
    print("[impostos] Migrando para o formato correto...")

    # Renomeia a tabela antiga
    cur.execute("ALTER TABLE cadastro_impostos RENAME TO cadastro_impostos_old")

    # Cria nova tabela no formato correto
    cur.execute("""
        CREATE TABLE cadastro_impostos (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            nome TEXT NOT NULL,
            origem TEXT CHECK(origem IN ('Nacional','Estadual','Municipal')) NOT NULL,
            valor REAL NOT NULL,
            dia_vencimento INTEGER NOT NULL,
            ativo INTEGER DEFAULT 1
        )
    """)
    conn.commit()

    # Lê dados antigos como dict (sem referenciar colunas inexistentes em SQL)
    conn.row_factory = sqlite3.Row
    rcur = conn.cursor()
    try:
        rcur.execute("SELECT * FROM cadastro_impostos_old")
        rows = rcur.fetchall()
    except sqlite3.OperationalError:
        rows = []

    # Descobre quais colunas existiam
    old_cols = set()
    if rows:
        old_cols = set(rows[0].keys())
    else:
        # se não há linhas, pega do PRAGMA
        old_cols = set(get_table_columns(conn, "cadastro_impostos_old"))

    # Prepara insert na nova tabela
    icur = conn.cursor()
    inserted = 0
    for row in rows:
        # Campos de origem (com defaults seguros)
        nome = row["nome"] if "nome" in old_cols else "Imposto"
        valor = float(row["valor"]) if "valor" in old_cols and row["valor"] is not None else 0.0

        # origem: tenta manter se já existia, senão default 'Nacional'
        origem = row["origem"] if "origem" in old_cols and row["origem"] else "Nacional"
        if origem not in ("Nacional", "Estadual", "Municipal"):
            origem = "Nacional"

        # dia_vencimento: se já existia a coluna dia_vencimento usa;
        # senão tenta derivar de 'vencimento' (se texto AAAA-MM-DD pega dia; se int usa direto); senão 20
        dia_v = 20
        if "dia_vencimento" in old_cols and row["dia_vencimento"] is not None:
            try:
                dia_v = int(row["dia_vencimento"])
            except:
                dia_v = 20
        elif "vencimento" in old_cols and row["vencimento"] is not None:
            v = row["vencimento"]
            if isinstance(v, int):
                dia_v = max(1, min(31, v))
            elif isinstance(v, str) and len(v) >= 8:
                # tenta pegar os 2 últimos dígitos (dia) de 'YYYY-MM-DD'
                try:
                    dia_v = int(v[-2:])
                    dia_v = max(1, min(31, dia_v))
                except:
                    dia_v = 20

        # ativo: se existir coluna 'ativo', usa; se existir 'pago', mapeia 1; caso contrário 1
        if "ativo" in old_cols and row["ativo"] is not None:
            ativo = 1 if int(row["ativo"]) == 1 else 0
        else:
            ativo = 1

        icur.execute("""
            INSERT INTO cadastro_impostos (nome, origem, valor, dia_vencimento, ativo)
            VALUES (?, ?, ?, ?, ?)
        """, (nome, origem, valor, dia_v, ativo))
        inserted += 1

    conn.commit()

    # Remove a tabela antiga
    cur.execute("DROP TABLE cadastro_impostos_old")
    conn.commit()
    print(f"[impostos] Migração concluída. Registros migrados: {inserted}")

def ensure_impostos_competencia(conn):
    cur = conn.cursor()
    cur.execute("""
        CREATE TABLE IF NOT EXISTS impostos_competencia (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            imposto_id INTEGER NOT NULL,
            competencia TEXT NOT NULL, -- MM/YYYY
            valor REAL NOT NULL,
            vencimento DATE NOT NULL,  -- YYYY-MM-DD
            pago INTEGER DEFAULT 0,
            FOREIGN KEY(imposto_id) REFERENCES cadastro_impostos(id) ON DELETE CASCADE
        )
    """)
    conn.commit()


# ----------------------------
# Inicialização principal
# ----------------------------
def inicializar_banco():
    conn = sqlite3.connect(CAMINHO_BANCO)
    conn.execute("PRAGMA foreign_keys = ON;")  # integridade referencial
    cursor = conn.cursor()

    # -------- TABELAS JÁ EXISTENTES --------
    cursor.execute("""
        CREATE TABLE IF NOT EXISTS registros_financeiros (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            cod_registro TEXT,
            data_exame TEXT,
            nome TEXT,
            empresa TEXT,
            convenio TEXT,
            procedimento TEXT,
            tipo_exame TEXT,
            medico TEXT,
            valor_convenio REAL,
            valor_medico REAL,
            competencia TEXT,
            data_cadastro TEXT,
            data_recebimento TEXT,
            duplicado INTEGER DEFAULT 0
        )
    """)

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS vendedores (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            nome TEXT,
            comissao REAL
        )
    """)

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS vendedor_convenio (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            vendedor_id INTEGER,
            convenio TEXT,
            FOREIGN KEY(vendedor_id) REFERENCES vendedores(id) ON DELETE CASCADE
        )
    """)

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS medicos (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            nome TEXT,
            crm TEXT,
            rqe TEXT,
            especialidade TEXT
        )
    """)

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS medico_procedimento (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            medico_id INTEGER,
            convenio TEXT,
            procedimento TEXT,
            valor REAL,
            FOREIGN KEY(medico_id) REFERENCES medicos(id) ON DELETE CASCADE
        )
    """)

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS convenios (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            nome TEXT,
            cnpj TEXT,
            email TEXT,
            telefone TEXT,
            logradouro TEXT,
            numero TEXT,
            bairro TEXT,
            complemento TEXT,
            cidade TEXT,
            estado TEXT,
            cep TEXT,
            responsavel TEXT,
            observacoes TEXT
            -- dia_vencimento é adicionado abaixo se não existir
        )
    """)

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS prestadores (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            nome TEXT,
            documento TEXT,
            servico TEXT,
            valor REAL,
            status TEXT
        )
    """)

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS funcionarios (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            nome TEXT,
            cargo TEXT,
            salario REAL,
            data_admissao TEXT,
            status TEXT
        )
    """)

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS pagamentos_prestadores (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            prestador_id INTEGER,
            competencia TEXT,
            valor_pago REAL,
            status TEXT,
            FOREIGN KEY(prestador_id) REFERENCES prestadores(id) ON DELETE CASCADE
        )
    """)

    cursor.execute("""
        CREATE TABLE IF NOT EXISTS notas_emitidas (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            numero TEXT,
            data_emissao TEXT,
            valor REAL,
            competencia TEXT
        )
    """)

    # -------- GARANTE/ATUALIZA ESQUEMAS --------
    add_column_if_not_exists(conn, "registros_financeiros", "Tipo de Exame", '"Tipo de Exame" TEXT')
    add_column_if_not_exists(conn, "convenios", "dia_vencimento", "dia_vencimento INTEGER")

    # Impostos (robusto)
    ensure_impostos_schema(conn)
    ensure_impostos_competencia(conn)

    conn.commit()
    conn.close()
    print("Banco atualizado com sucesso.")


if __name__ == "__main__":
    inicializar_banco()
