from PyQt5.QtWidgets import (
    QDialog, QVBoxLayout, QLabel, QLineEdit, QPushButton,
    QHBoxLayout, QMessageBox
)
from PyQt5.QtCore import Qt
import sqlite3
import os

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

class DialogEditarProcedimentoMedico(QDialog):
    def __init__(self, proc_id, parent=None):
        super().__init__(parent)
        self.proc_id = proc_id
        self.setWindowTitle("Editar Procedimento")
        self.setMinimumWidth(400)

        self.layout = QVBoxLayout(self)

        # Campos
        self.input_convenio = QLineEdit()
        self.input_procedimento = QLineEdit()
        self.input_valor = QLineEdit()
        self.input_valor.setAlignment(Qt.AlignCenter)

        self.layout.addWidget(QLabel("Convênio:"))
        self.layout.addWidget(self.input_convenio)
        self.layout.addWidget(QLabel("Procedimento:"))
        self.layout.addWidget(self.input_procedimento)
        self.layout.addWidget(QLabel("Valor Médico (R$):"))
        self.layout.addWidget(self.input_valor)

        # Botões de ação
        botoes = QHBoxLayout()
        btn_salvar = QPushButton("💾 Salvar")
        btn_excluir = QPushButton("🗑️ Excluir")
        btn_cancelar = QPushButton("Cancelar")

        botoes.addWidget(btn_salvar)
        botoes.addWidget(btn_excluir)
        botoes.addWidget(btn_cancelar)
        self.layout.addLayout(botoes)

        btn_salvar.clicked.connect(self.salvar)
        btn_excluir.clicked.connect(self.excluir)
        btn_cancelar.clicked.connect(self.reject)

        self.carregar_dados()

    def carregar_dados(self):
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("""
                SELECT convenio, procedimento, valor_medico
                FROM medico_procedimentos WHERE id = ?
            """, (self.proc_id,))
            resultado = cursor.fetchone()
            conn.close()

            if resultado:
                self.input_convenio.setText(str(resultado[0]))
                self.input_procedimento.setText(str(resultado[1]))
                self.input_valor.setText(str(resultado[2]).replace(".", ","))
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar dados: {e}")

    def salvar(self):
        convenio = self.input_convenio.text().strip()
        procedimento = self.input_procedimento.text().strip()
        valor = self.input_valor.text().strip().replace(",", ".")

        if not convenio or not procedimento or not valor:
            QMessageBox.warning(self, "Atenção", "Preencha todos os campos.")
            return

        try:
            valor_float = float(valor)
        except ValueError:
            QMessageBox.warning(self, "Erro", "Valor Médico inválido.")
            return

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("""
                UPDATE medico_procedimentos
                SET convenio = ?, procedimento = ?, valor_medico = ?
                WHERE id = ?
            """, (convenio, procedimento, valor_float, self.proc_id))
            conn.commit()
            conn.close()

            QMessageBox.information(self, "Sucesso", "Procedimento atualizado com sucesso.")
            self.accept()
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao salvar: {e}")

    def excluir(self):
        confirm = QMessageBox.question(
            self,
            "Confirmação",
            "Tem certeza que deseja excluir este procedimento?",
            QMessageBox.Yes | QMessageBox.No
        )
        if confirm == QMessageBox.Yes:
            try:
                conn = sqlite3.connect(CAMINHO_BANCO)
                cursor = conn.cursor()
                cursor.execute("DELETE FROM medico_procedimentos WHERE id = ?", (self.proc_id,))
                conn.commit()
                conn.close()

                QMessageBox.information(self, "Sucesso", "Procedimento excluído com sucesso.")
                self.accept()
            except Exception as e:
                QMessageBox.critical(self, "Erro", f"Erro ao excluir: {e}")
