from PyQt5.QtWidgets import (
    QDialog, QVBoxLayout, QFormLayout, QComboBox,
    QLineEdit, QPushButton, QMessageBox, QCompleter
)
from PyQt5.QtCore import Qt
import sqlite3
import os

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

class DialogIncluirProcedimento(QDialog):
    def __init__(self, medico_nome, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Incluir Procedimento ao Médico")
        self.setMinimumWidth(400)

        self.medico_nome = medico_nome
        layout = QVBoxLayout(self)
        form_layout = QFormLayout()

        # Combo Convênio com busca
        self.combo_convenio = QComboBox()
        convenios = self.buscar_convenios()
        self.combo_convenio.addItems(convenios)
        self.combo_convenio.setEditable(True)
        completer_conv = QCompleter(convenios, self)
        completer_conv.setCaseSensitivity(False)
        self.combo_convenio.setCompleter(completer_conv)

        # Combo Procedimento com busca
        self.combo_procedimento = QComboBox()
        procedimentos = self.buscar_procedimentos()
        self.combo_procedimento.addItems(procedimentos)
        self.combo_procedimento.setEditable(True)
        completer_proc = QCompleter(procedimentos, self)
        completer_proc.setCaseSensitivity(False)
        self.combo_procedimento.setCompleter(completer_proc)

        # Valor Médico
        self.input_valor = QLineEdit()
        self.input_valor.setPlaceholderText("Ex: 150.00")
        self.input_valor.setAlignment(Qt.AlignCenter)

        form_layout.addRow("Convênio:", self.combo_convenio)
        form_layout.addRow("Procedimento:", self.combo_procedimento)
        form_layout.addRow("Valor Médico (R$):", self.input_valor)

        self.btn_salvar = QPushButton("Salvar Conexão")
        self.btn_salvar.clicked.connect(self.salvar_conexao)

        layout.addLayout(form_layout)
        layout.addWidget(self.btn_salvar)

        self.input_valor.setFocus()

    def buscar_convenios(self):
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("""
                SELECT DISTINCT convenio 
                FROM registros_financeiros 
                WHERE convenio IS NOT NULL AND convenio != '' 
                ORDER BY convenio
            """)
            dados = [row[0] for row in cursor.fetchall()]
            conn.close()
            return dados
        except Exception as e:
            print(f"[ERRO] ao buscar convênios: {e}")
            return []

    def buscar_procedimentos(self):
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("""
                SELECT DISTINCT procedimento 
                FROM registros_financeiros 
                WHERE procedimento IS NOT NULL AND procedimento != '' 
                ORDER BY procedimento
            """)
            dados = [row[0] for row in cursor.fetchall()]
            conn.close()
            return dados
        except Exception as e:
            print(f"[ERRO] ao buscar procedimentos: {e}")
            return []

    def salvar_conexao(self):
        convenio = self.combo_convenio.currentText().strip()
        procedimento = self.combo_procedimento.currentText().strip()
        valor = self.input_valor.text().strip().replace(",", ".")

        if not convenio or not procedimento or not valor:
            QMessageBox.warning(self, "Atenção", "Todos os campos são obrigatórios.")
            return

        try:
            valor_float = float(valor)
        except ValueError:
            QMessageBox.warning(self, "Erro", "Valor inválido. Use ponto ou vírgula como separador decimal.")
            return

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()

            # Verifica ID do médico
            cursor.execute("SELECT id FROM medicos WHERE nome = ?", (self.medico_nome,))
            resultado = cursor.fetchone()
            if not resultado:
                QMessageBox.critical(self, "Erro", f"Médico '{self.medico_nome}' não encontrado.")
                return

            medico_id = resultado[0]

            # Verifica duplicidade
            cursor.execute("""
                SELECT 1 FROM medico_procedimentos
                WHERE medico_id = ? AND convenio = ? AND procedimento = ?
            """, (medico_id, convenio, procedimento))
            if cursor.fetchone():
                QMessageBox.warning(self, "Atenção", "Este procedimento já está vinculado ao médico.")
                return

            # Insere nova conexão
            cursor.execute("""
                INSERT INTO medico_procedimentos (medico_id, convenio, procedimento, valor_medico)
                VALUES (?, ?, ?, ?)
            """, (medico_id, convenio, procedimento, valor_float))
            conn.commit()
            conn.close()

            QMessageBox.information(self, "Sucesso", "Procedimento vinculado com sucesso.")
            self.accept()

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao salvar no banco: {e}")
