from PyQt5.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, QPushButton, QTableWidget,
    QTableWidgetItem, QHeaderView, QMessageBox
)
from PyQt5.QtCore import Qt
import sqlite3
import os

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

class DialogVincularConvenios(QDialog):
    def __init__(self, nome_vendedor, parent=None):
        super().__init__(parent)
        self.setWindowTitle(f"Vincular Convênios - {nome_vendedor}")
        self.setMinimumWidth(500)
        self.nome_vendedor = nome_vendedor

        layout = QVBoxLayout(self)

        layout.addWidget(QLabel("Marque os convênios que pertencem a este vendedor:"))

        self.tabela = QTableWidget()
        self.tabela.setColumnCount(2)
        self.tabela.setHorizontalHeaderLabels(["Convênio", "Vinculado"])
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        layout.addWidget(self.tabela)

        botoes_layout = QHBoxLayout()
        self.btn_salvar = QPushButton("Salvar")
        self.btn_cancelar = QPushButton("Cancelar")
        self.btn_salvar.clicked.connect(self.salvar_vinculacoes)
        self.btn_cancelar.clicked.connect(self.reject)
        botoes_layout.addStretch()
        botoes_layout.addWidget(self.btn_salvar)
        botoes_layout.addWidget(self.btn_cancelar)
        layout.addLayout(botoes_layout)

        self.carregar_convenios()

    def carregar_convenios(self):
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()

            # Obter ID do vendedor
            cursor.execute("SELECT id FROM vendedores WHERE nome = ?", (self.nome_vendedor,))
            resultado = cursor.fetchone()
            if not resultado:
                raise Exception("Vendedor não encontrado.")
            self.id_vendedor = resultado[0]

            # Convênios únicos dos registros financeiros
            cursor.execute("SELECT DISTINCT Convenio FROM registros_financeiros WHERE Convenio IS NOT NULL AND TRIM(Convenio) != '' ORDER BY Convenio")
            convenios = [row[0] for row in cursor.fetchall()]

            # Convênios já vinculados
            cursor.execute("SELECT nome_convenio FROM vendedor_convenio WHERE id_vendedor = ?", (self.id_vendedor,))
            vinculados = {row[0] for row in cursor.fetchall()}

            self.tabela.setRowCount(len(convenios))
            for i, convenio in enumerate(convenios):
                item_nome = QTableWidgetItem(convenio)
                item_nome.setFlags(Qt.ItemIsSelectable | Qt.ItemIsEnabled)
                self.tabela.setItem(i, 0, item_nome)

                item_checkbox = QTableWidgetItem()
                item_checkbox.setFlags(Qt.ItemIsUserCheckable | Qt.ItemIsEnabled)
                item_checkbox.setCheckState(Qt.Checked if convenio in vinculados else Qt.Unchecked)
                self.tabela.setItem(i, 1, item_checkbox)

            conn.close()

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar convênios:\n{e}")

    def salvar_vinculacoes(self):
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()

            cursor.execute("""
                CREATE TABLE IF NOT EXISTS vendedor_convenio (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    id_vendedor INTEGER NOT NULL,
                    nome_convenio TEXT NOT NULL,
                    FOREIGN KEY(id_vendedor) REFERENCES vendedores(id)
                )
            """)

            # Limpar os vínculos existentes
            cursor.execute("DELETE FROM vendedor_convenio WHERE id_vendedor = ?", (self.id_vendedor,))

            for i in range(self.tabela.rowCount()):
                nome_convenio = self.tabela.item(i, 0).text()
                check_state = self.tabela.item(i, 1).checkState()
                if check_state == Qt.Checked:
                    cursor.execute("""
                        INSERT INTO vendedor_convenio (id_vendedor, nome_convenio)
                        VALUES (?, ?)
                    """, (self.id_vendedor, nome_convenio))

            conn.commit()
            conn.close()
            QMessageBox.information(self, "Sucesso", "Vínculos atualizados com sucesso.")
            self.accept()

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao salvar vínculos:\n{e}")
