import sqlite3
import os
from datetime import datetime
import xml.etree.ElementTree as ET
import pandas as pd
import unicodedata
from scripts.gerar_xml_nfse import gerar_xml_nfse_por_convenio  # ajuste o path se necessário



CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

def tratar_duplicados(df):
    df["Duplicado"] = False
    df["NaoDuplicadoOIT"] = False

    if df.empty or "Nome" not in df.columns or "Procedimento" not in df.columns:
        return df

    oit_df = df[df["Procedimento"].str.upper().str.contains("RX - TÓRAX OIT - 2 ASSINATURAS", na=False)].copy()
    restantes_df = df[~df.index.isin(oit_df.index)].copy()

    grupo_oit = oit_df.groupby(["Nome", "Procedimento"])

    for (nome, proc), grupo in grupo_oit:
        medicos = grupo["Médico"].nunique()
        indices = grupo.index.tolist()

        if len(indices) == 2 and medicos == 2:
            oit_df.loc[indices, "NaoDuplicadoOIT"] = True
        else:
            oit_df.loc[indices[2:], "Duplicado"] = True

    colunas_chave = ["Nome", "Empresa", "Tipo de Exame", "Procedimento", "Data Exame"]
    if all(c in restantes_df.columns for c in colunas_chave):
        duplicatas = restantes_df.duplicated(subset=colunas_chave, keep='first')
        restantes_df.loc[duplicatas, "Duplicado"] = True

    return pd.concat([oit_df, restantes_df]).sort_index()



def gerar_xml_nfse_por_convenio(id_convenio, competencia, caminho_saida="nota.xml"):
    conn = sqlite3.connect(CAMINHO_BANCO)
    cursor = conn.cursor()

    # 🔹 Convênio
    cursor.execute("SELECT * FROM convenios WHERE id = ?", (id_convenio,))
    row_convenio = cursor.fetchone()
    if not row_convenio:
        print("❌ Convênio não encontrado.")
        return
    cols_convenio = [col[0] for col in cursor.description]
    tomador = dict(zip(cols_convenio, row_convenio))
    nome_convenio = tomador["nome"]

    # 🔹 Registros Financeiros
    df = pd.read_sql_query(
        "SELECT * FROM registros_financeiros WHERE Convenio = ? AND Competência = ?",
        conn, params=(nome_convenio, competencia)
    )
    df.columns = df.columns.str.normalize('NFKD').str.encode('ascii', errors='ignore').str.decode('utf-8').str.strip()
    df = df.loc[:, ~df.columns.duplicated()]

    if df.empty:
        print("❌ Nenhum registro financeiro encontrado.")
        return
    
    df = tratar_duplicados(df)
    df = df[df["Duplicado"] == False]
    valor_total = df["Valor Convenio"].sum()

    # 🔹 Prestadora
    cursor.execute("SELECT * FROM medical_laudos LIMIT 1")
    row_prestadora = cursor.fetchone()
    if not row_prestadora:
        print("❌ Dados da prestadora não encontrados.")
        return
    cols_prestadora = [col[0] for col in cursor.description]
    prestador = dict(zip(cols_prestadora, row_prestadora))

    conn.close()

    # 🔹 Geração do XML
    ns = "http://www.abrasf.org.br/nfse.xsd"
    ET.register_namespace('', ns)
    root = ET.Element(f"{{{ns}}}EnviarLoteRpsEnvio")
    lote_rps = ET.SubElement(root, "LoteRps", Id="lote1", versao="2.03")

    ET.SubElement(lote_rps, "NumeroLote").text = "1"
    cpfcnpj = ET.SubElement(lote_rps, "CpfCnpj")
    ET.SubElement(cpfcnpj, "Cnpj").text = ''.join(filter(str.isdigit, prestador["cnpj"]))
    ET.SubElement(lote_rps, "InscricaoMunicipal").text = ''.join(filter(str.isdigit, prestador["inscricao_municipal"]))

    ET.SubElement(lote_rps, "QuantidadeRps").text = "1"

    lista_rps = ET.SubElement(lote_rps, "ListaRps")
    rps = ET.SubElement(lista_rps, "Rps")
    inf_rps = ET.SubElement(rps, "InfRps", Id="rps1")

    identificacao_rps = ET.SubElement(inf_rps, "IdentificacaoRps")
    ET.SubElement(identificacao_rps, "Numero").text = f"{id_convenio}-{competencia.replace('/', '-')}"
    ET.SubElement(identificacao_rps, "Serie").text = "BH"
    ET.SubElement(identificacao_rps, "Tipo").text = "1"

    ET.SubElement(inf_rps, "DataEmissao").text = f"{datetime.now().isoformat()}"
    ET.SubElement(inf_rps, "NaturezaOperacao").text = "1"
    ET.SubElement(inf_rps, "RegimeEspecialTributacao").text = "1"
    ET.SubElement(inf_rps, "OptanteSimplesNacional").text = "1"
    ET.SubElement(inf_rps, "IncentivadorCultural").text = "2"
    ET.SubElement(inf_rps, "Status").text = "1"

    servico = ET.SubElement(inf_rps, "Servico")
    valores = ET.SubElement(servico, "Valores")
    def formatar_decimal(valor):
        return f"{valor:.2f}".replace(",", ".")

    ET.SubElement(valores, "ValorServicos").text = formatar_decimal(valor_total)
    ET.SubElement(valores, "IssRetido").text = "2"
    ET.SubElement(valores, "ValorIss").text = formatar_decimal(0)
    ET.SubElement(valores, "ValorPis").text = formatar_decimal(0)
    ET.SubElement(valores, "ValorCofins").text = formatar_decimal(0)
    ET.SubElement(valores, "ValorInss").text = formatar_decimal(0)
    ET.SubElement(valores, "ValorIr").text = formatar_decimal(0)
    ET.SubElement(valores, "ValorCsll").text = formatar_decimal(0)
    ET.SubElement(valores, "OutrasRetencoes").text = formatar_decimal(0)
    ET.SubElement(valores, "DescontoIncondicionado").text = formatar_decimal(0)
    ET.SubElement(valores, "DescontoCondicionado").text = formatar_decimal(0)


    ET.SubElement(servico, "ItemListaServico").text = "7.02"
    ET.SubElement(servico, "CodigoTributacaoMunicipio").text = "070201"
    ET.SubElement(servico, "Discriminacao").text = prestador.get("discriminacao", "").strip()
    ET.SubElement(servico, "CodigoMunicipio").text = "3106200"

    prestador_tag = ET.SubElement(inf_rps, "Prestador")
    ET.SubElement(prestador_tag, "Cnpj").text = ''.join(filter(str.isdigit, prestador["cnpj"]))
    ET.SubElement(prestador_tag, "InscricaoMunicipal").text = ''.join(filter(str.isdigit, prestador["inscricao_municipal"]))



    tomador_tag = ET.SubElement(inf_rps, "Tomador")
    id_tomador = ET.SubElement(tomador_tag, "IdentificacaoTomador")
    cpfcnpj_tomador = ET.SubElement(id_tomador, "CpfCnpj")
    ET.SubElement(cpfcnpj_tomador, "Cnpj").text = ''.join(filter(str.isdigit, tomador["cnpj"]))


    ET.SubElement(tomador_tag, "RazaoSocial").text = tomador.get("razao_social", "").strip()

    endereco = ET.SubElement(tomador_tag, "Endereco")
    ET.SubElement(endereco, "Endereco").text = tomador.get("logradouro", "").strip()
    ET.SubElement(endereco, "Numero").text = tomador.get("numero", "").strip()
    ET.SubElement(endereco, "Bairro").text = tomador.get("bairro", "").strip()
    ET.SubElement(endereco, "CodigoMunicipio").text = "3106200"
    ET.SubElement(endereco, "Uf").text = tomador.get("uf", "").strip()
    ET.SubElement(endereco, "Cep").text = ''.join(filter(str.isdigit, tomador.get("cep", "")))

    contato = ET.SubElement(tomador_tag, "Contato")
    ET.SubElement(contato, "Telefone").text = tomador.get("telefone", "").strip()
    ET.SubElement(contato, "Email").text = tomador.get("email", "").strip()


    # 🔚 Salvar XML
    tree = ET.ElementTree(root)
    tree.write(caminho_saida, encoding="utf-8", xml_declaration=True)

    print(f"✅ XML gerado com sucesso: {caminho_saida}")


# Teste direto
if __name__ == "__main__":
    gerar_xml_nfse_por_convenio(id_convenio=4, competencia="07/2025")
