# -*- coding: utf-8 -*-
from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QPushButton, QTableWidget,
    QTableWidgetItem, QDialog, QFormLayout, QLineEdit,
    QMessageBox, QHeaderView, QLabel, QFrame, QAbstractItemView,
    QComboBox, QSpinBox
)
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QDoubleValidator
import sqlite3, os
from datetime import datetime

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")
ORIGENS_PADRAO = ["Nacional", "Estadual", "Municipal"]

# ---------------- Schema & helpers ----------------
def _col_exists(cur, table, col):
    cur.execute(f"PRAGMA table_info({table})")
    return any(r[1] == col for r in cur.fetchall())

def _ensure_column(cur, table, col, decl, default_sql=None, log_table=None):
    if not _col_exists(cur, table, col):
        cur.execute(f"ALTER TABLE {table} ADD COLUMN {col} {decl}")
        if default_sql is not None:
            cur.execute(f"UPDATE {table} SET {col} = {default_sql}")
        if log_table:
            print(f"Coluna '{col}' criada em {log_table}.")
    else:
        if log_table:
            print(f"Coluna '{col}' já existe em {log_table}.")

def ensure_schema_outros_custos():
    with sqlite3.connect(CAMINHO_BANCO) as con:
        con.execute("PRAGMA foreign_keys=ON")
        cur = con.cursor()

        # 1) Tabela base (se não existir, cria já com colunas novas)
        cur.execute("""
            CREATE TABLE IF NOT EXISTS cadastro_outros_custos (
                id            INTEGER PRIMARY KEY AUTOINCREMENT,
                nome          TEXT NOT NULL,
                origem        TEXT,
                valor         REAL NOT NULL DEFAULT 0,
                ativo         INTEGER NOT NULL DEFAULT 1,
                periodicidade TEXT NOT NULL DEFAULT 'Mensal',  -- 'Mensal' | 'Anual'
                dia_venc      INTEGER,                         -- sempre usado
                mes_venc      INTEGER,                         -- só no Anual
                vig_ini       TEXT,                            -- 'YYYY-MM'
                vig_fim       TEXT,                            -- 'YYYY-MM' | NULL
                dia_semana    INTEGER                          -- compatibilidade (não usado)
            )
        """)
        cur.execute("CREATE INDEX IF NOT EXISTS idx_cad_outros_custos_nome ON cadastro_outros_custos(nome)")

        # 2) Migração de colunas caso a tabela já existisse com layout antigo
        table = "cadastro_outros_custos"
        _ensure_column(cur, table, "ativo",         "INTEGER NOT NULL DEFAULT 1",   default_sql="1",            log_table=table)
        _ensure_column(cur, table, "periodicidade", "TEXT NOT NULL DEFAULT 'Mensal'", default_sql="'Mensal'",   log_table=table)
        _ensure_column(cur, table, "dia_venc",      "INTEGER",                      default_sql="20",           log_table=table)
        _ensure_column(cur, table, "mes_venc",      "INTEGER",                      default_sql="NULL",         log_table=table)
        # vigência padrão: início = ano atual-mês atual; fim = NULL
        ano_atual = datetime.now().year
        mes_atual = datetime.now().month
        _ensure_column(cur, table, "vig_ini",       "TEXT",                         default_sql=f"'{ano_atual}-{mes_atual:02d}'", log_table=table)
        _ensure_column(cur, table, "vig_fim",       "TEXT",                         default_sql="NULL",         log_table=table)
        _ensure_column(cur, table, "dia_semana",    "INTEGER",                      default_sql="NULL",         log_table=table)

        con.commit()

    print(f"Schema verificado/ajustado em: {os.path.abspath(CAMINHO_BANCO)}")


# ---------- Helpers ----------
def _fmt_money(v: float) -> str:
    s = f"{float(v):,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
    return f"R$ {s}"

def _parse_money(txt: str) -> float:
    t = (txt or "").strip()
    if not t:
        return 0.0
    t = t.replace("R$", "").strip()
    t = t.replace(".", "").replace(",", ".")
    try:
        return float(t)
    except ValueError:
        return 0.0

def _meses_str():
    return [f"{m:02d}" for m in range(1, 13)]

def _anos_str(inicio=2025, fim=2031):
    return [str(a) for a in range(inicio, fim+1)]


# ---------- Dialog ----------
class DialogCadastroOutroCusto(QDialog):
    """
    Segue o mesmo UX do DialogCadastroImposto:
    - Periodicidade: Mensal/Anual
    - Dia venc. (sempre)
    - Mês venc. (se Anual)
    - Valor padrão, Origem, Status Ativo/Inativo
    - Vigência início e fim (fim opcional)
    """
    def __init__(self, registro=None, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Cadastro de Outro Custo")
        self.setFixedSize(380, 360)

        layout = QFormLayout(self)

        # --- Campos base ---
        self.input_nome = QLineEdit()

        self.combo_origem = QComboBox()
        self.combo_origem.addItems(ORIGENS_PADRAO)

        # Periodicidade (Mensal/Anual)
        self.combo_periodicidade = QComboBox()
        self.combo_periodicidade.addItems(["Mensal", "Anual"])

        # Valor padrão
        self.input_valor = QLineEdit()
        self.input_valor.setPlaceholderText("0,00")
        self.input_valor.setValidator(QDoubleValidator(0.0, 1e12, 2))

        # Dia do vencimento (Mensal/Anual)
        self.input_vencimento = QSpinBox()
        self.input_vencimento.setRange(1, 31)
        self.input_vencimento.setValue(20)

        # Mês do vencimento (Anual)
        self.combo_mes_anual = QComboBox()
        self.combo_mes_anual.addItems(_meses_str())

        # Ativo
        self.combo_ativo = QComboBox()
        self.combo_ativo.addItems(["Ativo", "Inativo"])

        # Vigência (início obrigatório, fim opcional)
        self.combo_ini_mes = QComboBox(); self.combo_ini_mes.addItems(_meses_str())
        self.combo_ini_ano = QComboBox(); self.combo_ini_ano.addItems(_anos_str())
        self.combo_fim_mes = QComboBox(); self.combo_fim_mes.addItems(["--"] + _meses_str())
        self.combo_fim_ano = QComboBox(); self.combo_fim_ano.addItems(["----"] + _anos_str())

        # --- Montagem do formulário (uma vez só) ---
        layout.addRow("Nome:", self.input_nome)
        layout.addRow("Origem:", self.combo_origem)
        layout.addRow("Periodicidade:", self.combo_periodicidade)

        # Linhas dependentes da periodicidade
        self.row_mensal_anual = self._h(QLabel("Dia venc."), self.input_vencimento)  # sempre visível
        self.row_mes_anual    = self._h(QLabel("Mês venc."), self.combo_mes_anual)   # só no Anual
        layout.addRow(self.row_mensal_anual)
        layout.addRow(self.row_mes_anual)

        layout.addRow("Valor Padrão:", self.input_valor)
        layout.addRow("Status:", self.combo_ativo)
        layout.addRow("Aparecer a partir de:", self._h(self.combo_ini_mes, QLabel("/"), self.combo_ini_ano))
        layout.addRow("Parar em (opcional):", self._h(self.combo_fim_mes, QLabel("/"), self.combo_fim_ano))

        btn_salvar = QPushButton("Salvar")
        btn_salvar.clicked.connect(self.accept)
        layout.addRow(btn_salvar)

        # Mostrar/ocultar campos conforme periodicidade
        def _update_fields_by_periodicidade():
            per = self.combo_periodicidade.currentText()
            self.row_mensal_anual.setVisible(True)        # dia sempre usado
            self.row_mes_anual.setVisible(per == "Anual") # mês só no Anual
        self.combo_periodicidade.currentIndexChanged.connect(_update_fields_by_periodicidade)
        _update_fields_by_periodicidade()

        # Preenchimento em modo edição
        if registro:
            # Ordem dos campos vinda do SELECT no método editar():
            # (id, nome, origem, valor, dia_venc, ativo, vig_ini, vig_fim, periodicidade, dia_semana, mes_venc)
            self.input_nome.setText(registro[1] or "")
            origem_db = (registro[2] or "").strip()
            if origem_db and origem_db not in ORIGENS_PADRAO:
                self.combo_origem.addItem(origem_db)  # mantém compatibilidade com dados antigos
            self.combo_origem.setCurrentText(origem_db if origem_db else ORIGENS_PADRAO[0])

            self.input_valor.setText(f"{float(registro[3] or 0):.2f}".replace(".", ","))
            self.input_vencimento.setValue(int(registro[4]) if registro[4] else 20)
            self.combo_ativo.setCurrentIndex(0 if int(registro[5] or 1) == 1 else 1)

            # vig_ini
            vig_ini = (registro[6] or "")
            parts = vig_ini.split("-")
            if len(parts) == 2:
                self.combo_ini_ano.setCurrentText(parts[0])
                self.combo_ini_mes.setCurrentText(parts[1])

            # vig_fim
            vig_fim = (registro[7] or "")
            if vig_fim:
                parts = vig_fim.split("-")
                if len(parts) == 2:
                    self.combo_fim_ano.setCurrentText(parts[0])
                    self.combo_fim_mes.setCurrentText(parts[1])

            self.combo_periodicidade.setCurrentText(registro[8] or "Mensal")
            if registro[10] is not None:
                self.combo_mes_anual.setCurrentText(f"{int(registro[10]):02d}")
            _update_fields_by_periodicidade()

        else:
            # Valores iniciais amigáveis
            self.combo_ini_mes.setCurrentText(datetime.now().strftime("%m"))
            self.combo_ini_ano.setCurrentText(datetime.now().strftime("%Y"))

    def _h(self, *widgets):
        w = QWidget(); lay = QHBoxLayout(w); lay.setContentsMargins(0,0,0,0); lay.setSpacing(6)
        for wd in widgets: lay.addWidget(wd)
        lay.addStretch(1)
        return w

    def get_dados(self):
        valor = _parse_money(self.input_valor.text())

        nome = self.input_nome.text().strip()
        origem = self.combo_origem.currentText().strip()
        ativo = 1 if self.combo_ativo.currentText() == "Ativo" else 0

        per = self.combo_periodicidade.currentText().strip()  # 'Mensal' | 'Anual'
        dia_venc = int(self.input_vencimento.value())
        mes_anual = int(self.combo_mes_anual.currentText()) if per == "Anual" else None

        vig_ini = f"{self.combo_ini_ano.currentText()}-{self.combo_ini_mes.currentText()}"
        vig_fim = None
        fim_ano = self.combo_fim_ano.currentText().strip()
        fim_mes = self.combo_fim_mes.currentText().strip()
        if fim_ano and fim_mes and fim_ano != "----" and fim_mes != "--":
            vig_fim = f"{fim_ano}-{fim_mes}"

        dia_semana = None  # compatibilidade
        return (nome, origem, float(valor), dia_venc, ativo, vig_ini, vig_fim, per, dia_semana, mes_anual)


# ---------- Tela Principal (Cadastro) ----------
class ModuloCadastroOutrosCustos(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        ensure_schema_outros_custos()

        root = QVBoxLayout(self)
        root.setSpacing(20)
        root.setContentsMargins(20, 20, 20, 20)

        estilo_botao = """
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover { background-color: #2980b9; }
        """

        # Ações
        topbar = QHBoxLayout()
        self.btn_novo    = QPushButton("➕ Novo Custo")
        self.btn_editar  = QPushButton("✏️ Editar Custo")
        self.btn_excluir = QPushButton("🗑️ Excluir Custo")
        for b in (self.btn_novo, self.btn_editar, self.btn_excluir):
            b.setStyleSheet(estilo_botao)

        self.btn_novo.clicked.connect(self.novo)
        self.btn_editar.clicked.connect(self.editar)
        self.btn_excluir.clicked.connect(self.excluir)

        topbar.addWidget(self.btn_novo)
        topbar.addWidget(self.btn_editar)
        topbar.addWidget(self.btn_excluir)
        topbar.addStretch(1)
        root.addLayout(topbar)

        # Busca
        busca = QHBoxLayout()
        self.campo_busca = QLineEdit()
        self.campo_busca.setPlaceholderText("🔍 Buscar por nome / origem...")
        self.campo_busca.textChanged.connect(self.filtrar)
        busca.addWidget(self.campo_busca)
        root.addLayout(busca)

        # Divisor
        linha = QFrame()
        linha.setFrameShape(QFrame.HLine)
        linha.setFrameShadow(QFrame.Sunken)
        root.addWidget(linha)

        # Tabela (exibição ampliada)
        self.tabela = QTableWidget()
        self.tabela.setColumnCount(6)
        self.tabela.setHorizontalHeaderLabels(["Nome", "Origem", "Valor", "Periodicidade", "Vencimento", "Status"])
        self.tabela.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tabela.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tabela.setAlternatingRowColors(True)
        self.tabela.verticalHeader().setVisible(False)
        self.tabela.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.tabela.setSortingEnabled(True)

        header = self.tabela.horizontalHeader()
        header.setSectionResizeMode(QHeaderView.Stretch)
        header.setDefaultAlignment(Qt.AlignCenter)

        root.addWidget(self.tabela, 1)

        # Rodapé: total
        rodape = QHBoxLayout()
        rodape.addStretch(1)
        self.lbl_total = QLabel("Total: R$ 0,00")
        self.lbl_total.setAlignment(Qt.AlignRight)
        rodape.addWidget(self.lbl_total)
        root.addLayout(rodape)

        self.carregar()

    # --- Core ---
    def carregar(self):
        self.tabela.setRowCount(0)
        total = 0.0
        with sqlite3.connect(CAMINHO_BANCO) as con:
            cur = con.cursor()
            # Ordem dos campos casa com o diálogo de edição
            cur.execute("""
                SELECT id, nome, origem, valor, dia_venc, ativo, vig_ini, vig_fim, periodicidade, dia_semana, mes_venc
                  FROM cadastro_outros_custos
                 ORDER BY nome
            """)
            rows = cur.fetchall()

        for r, row in enumerate(rows):
            iid, nome, origem, valor, dia_venc, ativo, vig_ini, vig_fim, per, _, mes_venc = row
            self.tabela.insertRow(r)

            it_nome = QTableWidgetItem(nome or "")
            it_nome.setData(Qt.UserRole, iid)
            it_nome.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 0, it_nome)

            it_org = QTableWidgetItem(origem or "")
            it_org.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 1, it_org)

            it_val = QTableWidgetItem(_fmt_money(valor or 0))
            it_val.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 2, it_val)

            it_per = QTableWidgetItem(per or "Mensal")
            it_per.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 3, it_per)

            if (per or "Mensal") == "Anual":
                venc = f"{(dia_venc or 0):02d}/{(mes_venc or 0):02d}"
            else:
                venc = f"Todo dia {(dia_venc or 0):02d}"
            it_venc = QTableWidgetItem(venc)
            it_venc.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 4, it_venc)

            it_status = QTableWidgetItem("Ativo" if int(ativo or 1) == 1 else "Inativo")
            it_status.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 5, it_status)

            total += float(valor or 0)

        self.lbl_total.setText(f"Total: {_fmt_money(total)}")

    def filtrar(self):
        texto = (self.campo_busca.text() or "").strip().lower()
        termos = [t for t in texto.split() if t]
        for row in range(self.tabela.rowCount()):
            blob = " ".join(
                (self.tabela.item(row, c).text().lower() if self.tabela.item(row, c) else "")
                for c in range(self.tabela.columnCount())
            )
            visivel = all(t in blob for t in termos)
            self.tabela.setRowHidden(row, not visivel)

    def _get_selected_id(self):
        linha = self.tabela.currentRow()
        if linha < 0:
            return None
        it = self.tabela.item(linha, 0)
        return None if it is None else it.data(Qt.UserRole)

    # --- Ações ---
    def novo(self):
        dlg = DialogCadastroOutroCusto(parent=self)
        if dlg.exec_():
            nome, origem, valor, dia_venc, ativo, vig_ini, vig_fim, per, dia_semana, mes_anual = dlg.get_dados()
            if not nome:
                QMessageBox.warning(self, "Atenção", "Informe o nome do custo.")
                return
            with sqlite3.connect(CAMINHO_BANCO) as con:
                cur = con.cursor()
                cur.execute("""
                    INSERT INTO cadastro_outros_custos
                        (nome, origem, valor, dia_venc, ativo, vig_ini, vig_fim, periodicidade, dia_semana, mes_venc)
                    VALUES (?,?,?,?,?,?,?,?,?,?)
                """, (nome, origem, float(valor), dia_venc, int(ativo), vig_ini, vig_fim, per, dia_semana, mes_anual))
                con.commit()
            self.carregar()

    def editar(self):
        rid = self._get_selected_id()
        if rid is None:
            QMessageBox.warning(self, "Atenção", "Selecione um custo para editar.")
            return

        with sqlite3.connect(CAMINHO_BANCO) as con:
            cur = con.cursor()
            cur.execute("""
                SELECT id, nome, origem, valor, dia_venc, ativo, vig_ini, vig_fim, periodicidade, dia_semana, mes_venc
                  FROM cadastro_outros_custos
                 WHERE id=?
            """, (rid,))
            registro = cur.fetchone()

        dlg = DialogCadastroOutroCusto(registro, parent=self)
        if dlg.exec_():
            nome, origem, valor, dia_venc, ativo, vig_ini, vig_fim, per, dia_semana, mes_anual = dlg.get_dados()
            if not nome:
                QMessageBox.warning(self, "Atenção", "Informe o nome do custo.")
                return

            with sqlite3.connect(CAMINHO_BANCO) as con:
                cur = con.cursor()
                cur.execute("""
                    UPDATE cadastro_outros_custos
                       SET nome=?, origem=?, valor=?, dia_venc=?, ativo=?, vig_ini=?, vig_fim=?, periodicidade=?, dia_semana=?, mes_venc=?
                     WHERE id=?
                """, (nome, origem, float(valor), dia_venc, int(ativo), vig_ini, vig_fim, per, dia_semana, mes_anual, rid))
                con.commit()
            self.carregar()

    def excluir(self):
        rid = self._get_selected_id()
        if rid is None:
            QMessageBox.warning(self, "Atenção", "Selecione um custo para excluir.")
            return

        nome = self.tabela.item(self.tabela.currentRow(), 0).text() if self.tabela.currentRow() >= 0 else ""
        confirm = QMessageBox.question(
            self, "Confirmar", f"Deseja excluir '{nome}'?",
            QMessageBox.Yes | QMessageBox.No
        )
        if confirm == QMessageBox.Yes:
            with sqlite3.connect(CAMINHO_BANCO) as con:
                cur = con.cursor()
                cur.execute("DELETE FROM cadastro_outros_custos WHERE id=?", (rid,))
                con.commit()
            self.carregar()
