# -*- coding: utf-8 -*-
from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QPushButton, QTableWidget,
    QTableWidgetItem, QMessageBox, QLabel, QHeaderView, QFrame,
    QAbstractItemView, QLineEdit, QComboBox, QInputDialog
)
from PyQt5.QtCore import Qt
import sqlite3, os, calendar
from datetime import datetime, date
from gui.modulo10_cadastro_outros_custos import ensure_schema_outros_custos


CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

# ---------------- helpers ----------------

def _fmt_money(v: float) -> str:
    s = f"{float(v):,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
    return f"R$ {s}"

def _parse_money(txt: str) -> float:
    t = (txt or "").strip()
    if not t:
        return 0.0
    t = t.replace("R$", "").strip().replace(".", "").replace(",", ".")
    try:
        return float(t)
    except ValueError:
        return 0.0

def _meses_padrao():
    return [f"{m:02d}" for m in range(1, 13)]

def _anos_disponiveis():
    ano_atual = int(datetime.now().strftime("%Y"))
    try:
        with sqlite3.connect(CAMINHO_BANCO) as con:
            cur = con.cursor()
            cur.execute("SELECT MIN(vig_ini), MAX(COALESCE(vig_fim, vig_ini)) FROM cadastro_outros_custos")
            mn, mx = cur.fetchone() or (None, None)
        anos = set()
        for s in (mn, mx):
            if s and len(s) == 7 and s[4] == "-":
                anos.add(int(s[:4]))
        if anos:
            lo = min(anos) - 1
            hi = max(anos) + 1
            return [str(a) for a in range(lo, hi + 1)]
    except Exception:
        pass
    return [str(ano_atual), str(ano_atual + 1)]

def _safe_day(y: int, m: int, d: int) -> int:
    return min(d, calendar.monthrange(y, m)[1])

def _status_info(pago: int, due: date):
    if pago:
        return "🟢 Pago", "Pago"
    delta = (due - date.today()).days
    if delta < 0:
        return "🔴 Atrasado", "Atrasado"
    if delta <= 5:
        return "🟡 Próximo do Venc", "Próximo do Venc"
    return "⚪ Em aberto", "Em aberto"

# ---------------- widget ----------------
class ModuloInfoOutrosCustos(QWidget):
    """
    Informações/Competência para Outros Custos:
    - Busca por nome/origem
    - Seleção de Mês e Ano
    - Editar/Limpar Valor (override por competência, sem alterar cadastro)
    - Marcar como Pago/Não Pago
    - Filtra por vigência e ativos
    - Exibe Periodicidade, Vencimento, Valor, Status, Data Pag.
    """
    
    def _make_legend(self, color_hex: str, title: str):
        box = QFrame()
        lay = QVBoxLayout(box)
        lay.setContentsMargins(12, 6, 12, 6)
        lay.setSpacing(2)

        # Cabeçalho com "bolinha" colorida + título
        head = QHBoxLayout()
        head.setContentsMargins(0, 0, 0, 0)
        head.setSpacing(6)
        dot = QLabel("●")
        dot.setStyleSheet(f"color: {color_hex}; font-size: 16px;")
        title_lbl = QLabel(title); title_lbl.setObjectName("title")
        head.addWidget(dot)
        head.addWidget(title_lbl)
        head.addStretch(1)

        value_lbl = QLabel("R$ 0,00"); value_lbl.setObjectName("value")
        count_lbl = QLabel("0 itens"); count_lbl.setObjectName("count")

        lay.addLayout(head)
        lay.addWidget(value_lbl)
        lay.addWidget(count_lbl)

        return {"widget": box, "val": value_lbl, "count": count_lbl}

    
    def __init__(self, parent=None):
        super().__init__(parent)
        ensure_schema_outros_custos()

        root = QVBoxLayout(self)
        root.setSpacing(20)
        root.setContentsMargins(20, 20, 20, 20)

        estilo_botao = """
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover { background-color: #2980b9; }
        """

        # Top (Mês/Ano + ações)
        top = QHBoxLayout()

        top.addWidget(QLabel("Mês:"))
        self.combo_mes = QComboBox()
        self.combo_mes.addItems(_meses_padrao())
        self.combo_mes.setCurrentText(datetime.now().strftime("%m"))
        self.combo_mes.currentIndexChanged.connect(self.carregar)
        top.addWidget(self.combo_mes)

        top.addWidget(QLabel("Ano:"))
        self.combo_ano = QComboBox()
        self.combo_ano.addItems(_anos_disponiveis())
        self.combo_ano.setCurrentText(datetime.now().strftime("%Y"))
        self.combo_ano.currentIndexChanged.connect(self.carregar)
        top.addWidget(self.combo_ano)

        top.addStretch(1)

        self.btn_editar_valor = QPushButton("Editar Valor")
        self.btn_limpar_valor = QPushButton("Limpar Valor")
        self.btn_pagar        = QPushButton("Marcar como Pago")
        self.btn_nao_pagar    = QPushButton("Marcar como NÃO Pago")

        for b in (self.btn_editar_valor, self.btn_limpar_valor, self.btn_pagar, self.btn_nao_pagar):
            b.setStyleSheet(estilo_botao)

        self.btn_editar_valor.clicked.connect(self._definir_valor_competencia)
        self.btn_limpar_valor.clicked.connect(self._limpar_valor_competencia)
        self.btn_pagar.clicked.connect(lambda: self._set_pago(True))
        self.btn_nao_pagar.clicked.connect(lambda: self._set_pago(False))

        top.addWidget(self.btn_editar_valor)
        top.addWidget(self.btn_limpar_valor)
        top.addWidget(self.btn_pagar)
        top.addWidget(self.btn_nao_pagar)
        root.addLayout(top)

        # Busca
        busca = QHBoxLayout()
        self.campo_busca = QLineEdit()
        self.campo_busca.setPlaceholderText("🔍 Buscar por nome / origem...")
        self.campo_busca.textChanged.connect(self._filtrar_local)
        busca.addWidget(self.campo_busca)
        root.addLayout(busca)

        # Divisor
        linha = QFrame()
        linha.setFrameShape(QFrame.HLine)
        linha.setFrameShadow(QFrame.Sunken)
        root.addWidget(linha)

        # Tabela
        self.tabela = QTableWidget()
        self.tabela.setColumnCount(7)
        self.tabela.setHorizontalHeaderLabels([
            "Nome", "Origem", "Periodicidade", "Vencimento", "Valor", "Situação", "Data Pag."
        ])
        self.tabela.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tabela.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tabela.setAlternatingRowColors(True)
        self.tabela.verticalHeader().setVisible(False)
        self.tabela.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.tabela.setSortingEnabled(True)

        header = self.tabela.horizontalHeader()
        header.setSectionResizeMode(QHeaderView.Stretch)
        header.setDefaultAlignment(Qt.AlignCenter)

        root.addWidget(self.tabela, 1)

        # =========================
        # Rodapé com legendas BONITO
        # =========================
        self.footer_bar = QFrame()
        self.footer_bar.setObjectName("footerBar")
        self.footer_bar.setStyleSheet("""
            QFrame#footerBar {
                background: #fff;
                border-top: 1px solid #e9ecef;
                border-radius: 8px;
            }
            QLabel#value { font-weight: 700; }
            QLabel#title { color: #59636e; font-weight: 600; }
            QLabel#count { color: #7f8c8d; font-weight: 500; }
        """)

        footer_layout = QHBoxLayout(self.footer_bar)
        footer_layout.setContentsMargins(14, 10, 14, 10)
        footer_layout.setSpacing(12)

        # grupos: Total (cinza), Abertos (laranja), Pago (verde)
        self.grp_total  = self._make_legend("#34495e", "Total")
        self.grp_aberto = self._make_legend("#f39c12", "Em aberto / Próx. / Atrasado")
        self.grp_pago   = self._make_legend("#27ae60", "Pago")

        footer_layout.addWidget(self.grp_total["widget"])
        footer_layout.addStretch(1)
        footer_layout.addWidget(self.grp_aberto["widget"])
        footer_layout.addStretch(1)
        footer_layout.addWidget(self.grp_pago["widget"])

        root.addWidget(self.footer_bar)


        # Carregar
        self.carregar()

    # --------------------- Data ---------------------
    def _competencia_txt(self) -> str:
        return f"{self.combo_mes.currentText()}/{self.combo_ano.currentText()}"

    def _competencia_year_month(self):
        return int(self.combo_ano.currentText()), int(self.combo_mes.currentText())

    def carregar(self):
        comp_txt = self._competencia_txt()
        ano_sel, mes_sel = self._competencia_year_month()
        comp_ym = f"{ano_sel:04d}-{mes_sel:02d}"
        self.tabela.setRowCount(0)
        total = 0.0
        paid_sum = 0.0
        open_sum = 0.0
        paid_count = 0
        open_count = 0


        with sqlite3.connect(CAMINHO_BANCO) as con:
            con.row_factory = sqlite3.Row
            cur = con.cursor()
            # Ativos, dentro da vigência; Anual só aparece no mês mes_venc
            cur.execute("""
                SELECT
                    c.id               AS custo_id,
                    c.nome,
                    c.origem,
                    c.valor            AS valor_cadastro,
                    c.periodicidade,
                    c.dia_venc,
                    c.mes_venc,
                    c.ativo,
                    c.vig_ini,
                    c.vig_fim,
                    oc.valor           AS valor_override,
                    oc.pago,
                    oc.data_pagamento
                FROM cadastro_outros_custos c
                LEFT JOIN outros_custos_competencia oc
                       ON oc.custo_id = c.id
                      AND oc.competencia = ?
                WHERE c.ativo = 1
                  AND (c.vig_ini IS NULL OR c.vig_ini <= ?)
                  AND (c.vig_fim IS NULL OR c.vig_fim >= ?)
                  AND (
                        c.periodicidade = 'Mensal'
                        OR (c.periodicidade = 'Anual' AND printf('%02d', COALESCE(c.mes_venc, 1)) = ?)
                      )
                ORDER BY c.nome
            """, (comp_txt, comp_ym, comp_ym, f"{mes_sel:02d}"))
            rows = cur.fetchall()
            
            # === MATERIALIZAR VALORES EFETIVOS NA COMPETÊNCIA (garante soma no Faturamento) ===
            try:
                with sqlite3.connect(CAMINHO_BANCO) as con2:
                    cur2 = con2.cursor()
                    cur2.executemany("""
                        INSERT INTO outros_custos_competencia (custo_id, competencia, valor)
                        VALUES (?,?,?)
                        ON CONFLICT(custo_id, competencia) DO UPDATE SET
                            -- mantém override já gravado; se não houver, preenche com o default do cadastro
                            valor = COALESCE(outros_custos_competencia.valor, excluded.valor)
                    """, [
                        (
                            row["custo_id"],
                            comp_txt,
                            float(row["valor_override"] if row["valor_override"] is not None else (row["valor_cadastro"] or 0.0))
                        )
                        for row in rows
                    ])
                    con2.commit()
            except Exception as e:
                print("[Outros Custos] materialização falhou:", e)


        for r, row in enumerate(rows):
            self.tabela.insertRow(r)

            custo_id   = row["custo_id"]
            nome       = row["nome"] or ""
            origem     = row["origem"] or ""
            per        = (row["periodicidade"] or "Mensal")
            dia_venc   = int(row["dia_venc"] or 1)
            mes_venc   = int(row["mes_venc"] or mes_sel)
            v_cad      = float(row["valor_cadastro"] or 0)
            v_over     = row["valor_override"]
            pago       = int(row["pago"] or 0)
            dtpag      = row["data_pagamento"] or ""

            # Valor efetivo: override ou cadastro
            valor = float(v_over) if v_over is not None else v_cad
            total += valor

            # Vencimento + status
            m = mes_sel if per == "Mensal" else mes_venc
            d = _safe_day(ano_sel, m, dia_venc)
            due = date(ano_sel, m, d)
            status_cell, status_txt = _status_info(pago, due)
            
            # >>> SOMATÓRIOS DO RODAPÉ <<<
            if pago:
                paid_sum += valor
                paid_count += 1
            else:
                open_sum += valor
                open_count += 1

            venc_txt = (f"Todo dia {dia_venc:02d}" if per == "Mensal" else f"{dia_venc:02d}/{m:02d}")

            # ---- cells ----
            it_nome = QTableWidgetItem(nome)
            it_nome.setData(Qt.UserRole, custo_id)
            it_nome.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 0, it_nome)

            it_org = QTableWidgetItem(origem)
            it_org.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 1, it_org)

            it_per = QTableWidgetItem(per)
            it_per.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 2, it_per)

            it_venc = QTableWidgetItem(venc_txt)
            it_venc.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 3, it_venc)

            it_val = QTableWidgetItem(_fmt_money(valor))
            it_val.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 4, it_val)

            it_st = QTableWidgetItem(status_cell)   # agora: bolinha + texto (ex.: "🟢 Pago")
            it_st.setTextAlignment(Qt.AlignCenter)
            it_st.setToolTip(status_txt)
            self.tabela.setItem(r, 5, it_st)

            it_dt = QTableWidgetItem(dtpag)
            it_dt.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(r, 6, it_dt)

        # Atualiza legendas do rodapé
        self.grp_total["val"].setText(_fmt_money(total))
        self.grp_total["count"].setText(f"{paid_count + open_count} itens")

        self.grp_aberto["val"].setText(_fmt_money(open_sum))
        self.grp_aberto["count"].setText(f"{open_count} itens")

        self.grp_pago["val"].setText(_fmt_money(paid_sum))
        self.grp_pago["count"].setText(f"{paid_count} itens")

        self._filtrar_local()  # re-aplica busca


    def _get_selected(self):
        row = self.tabela.currentRow()
        if row < 0:
            return None
        it = self.tabela.item(row, 0)
        custo_id = it.data(Qt.UserRole) if it else None
        return (row, custo_id)

    # -------------------- Ações ---------------------
    def _set_pago(self, pago: bool):
        sel = self._get_selected()
        if not sel:
            QMessageBox.warning(self, "Atenção", "Selecione um custo na lista.")
            return
        row, custo_id = sel               # <-- precisamos do 'row' para ler o valor da tabela
        comp = self._competencia_txt()
        hoje = datetime.now().strftime("%d/%m/%Y")

        # valor efetivo que está visível na coluna "Valor" (coluna 4)
        try:
            valor_efetivo = _parse_money(self.tabela.item(row, 4).text())
        except Exception:
            valor_efetivo = 0.0

        with sqlite3.connect(CAMINHO_BANCO) as con:
            cur = con.cursor()
            cur.execute("""
                INSERT INTO outros_custos_competencia (custo_id, competencia, valor, pago, data_pagamento)
                VALUES (?,?,?,?,?)
                ON CONFLICT(custo_id, competencia) DO UPDATE SET
                    pago = excluded.pago,
                    data_pagamento = excluded.data_pagamento,
                    -- mantém override já gravado; se não houver, usa o valor atual da tela
                    valor = COALESCE(outros_custos_competencia.valor, excluded.valor)
            """, (custo_id, comp, float(valor_efetivo), 1 if pago else 0, hoje if pago else None))
            con.commit()

        self.carregar()


    def _definir_valor_competencia(self):
        sel = self._get_selected()
        if not sel:
            QMessageBox.warning(self, "Atenção", "Selecione um custo na lista.")
            return
        row, custo_id = sel
        comp = self._competencia_txt()

        valor_atual_txt = self.tabela.item(row, 4).text()  # "Valor"
        novo_txt, ok = QInputDialog.getText(self, "Valor", "Informe o valor (ex.: 1.234,56):", text=valor_atual_txt)
        if not ok:
            return
        novo_valor = _parse_money(novo_txt)

        with sqlite3.connect(CAMINHO_BANCO) as con:
            cur = con.cursor()
            cur.execute("""
                INSERT INTO outros_custos_competencia (custo_id, competencia, valor)
                VALUES (?,?,?)
                ON CONFLICT(custo_id, competencia) DO UPDATE SET
                    valor = excluded.valor
            """, (custo_id, comp, float(novo_valor)))
            con.commit()

        self.carregar()

    def _limpar_valor_competencia(self):
        sel = self._get_selected()
        if not sel:
            QMessageBox.warning(self, "Atenção", "Selecione um custo na lista.")
            return
        _, custo_id = sel
        comp = self._competencia_txt()

        # pega o valor padrão do cadastro
        with sqlite3.connect(CAMINHO_BANCO) as con:
            cur = con.cursor()
            cur.execute("SELECT COALESCE(valor, 0) FROM cadastro_outros_custos WHERE id = ?", (custo_id,))
            v_default = cur.fetchone()
            v_default = float(v_default[0] or 0.0) if v_default else 0.0

            cur.execute("""
                INSERT INTO outros_custos_competencia (custo_id, competencia, valor)
                VALUES (?,?,?)
                ON CONFLICT(custo_id, competencia) DO UPDATE SET
                    -- remove apenas o override; mantém linha da competência com o valor do cadastro
                    valor = excluded.valor
            """, (custo_id, comp, v_default))
            con.commit()

        self.carregar()


    # ------------------- Filtro local -------------------
    def _filtrar_local(self):
        texto = (self.campo_busca.text() or "").strip().lower()
        termos = [t for t in texto.split() if t]
        for r in range(self.tabela.rowCount()):
            blob = " ".join(
                (self.tabela.item(r, c).text().lower() if self.tabela.item(r, c) else "")
                for c in (0, 1)  # Nome/Origem
            )
            visivel = all(t in blob for t in termos)
            self.tabela.setRowHidden(r, not visivel)
