from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QLabel, QComboBox, QPushButton,
    QGridLayout, QFrame, QTableWidget, QTableWidgetItem, QHeaderView,
    QTabWidget, QMessageBox, QAbstractItemView, QFileDialog
)
from PyQt5.QtCore import Qt, QDate
from PyQt5.QtGui import QFont
import os, sqlite3
from datetime import datetime, date
import unicodedata
import re
from PyQt5.QtWidgets import QStyledItemDelegate

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

# ==============================
# Helpers de formatação
# ==============================

class CenterAlignDelegate(QStyledItemDelegate):
    def initStyleOption(self, option, index):
        super().initStyleOption(option, index)
        option.displayAlignment = Qt.AlignCenter

def fmt_money(v) -> str:
    try:
        n = float(v or 0.0)
    except Exception:
        n = 0.0
    s = f"{n:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
    return f"R$ {s}"


def safe_fetchone(cur):
    row = cur.fetchone()
    return row[0] if row and len(row) > 0 and row[0] is not None else 0.0


# ==============================
# Cartão KPI estilizado
# ==============================
class KpiCard(QFrame):
    def __init__(self, titulo: str, dica: str = ""):
        super().__init__()
        self.setObjectName("KpiCard")
        self.setStyleSheet(
            """
            QFrame#KpiCard {background: #f7f9fc; border: 1px solid #e6ecf2; border-radius: 12px;}
            QLabel[titulo="true"] {font-weight: 600; color: #1f2937;}
            QLabel[valor="true"] {font-size: 22px; font-weight: 700;}
            QLabel[sub="true"] {color: #6b7280; font-size: 11px;}
            """
        )
        lay = QVBoxLayout(self)
        lay.setContentsMargins(12, 10, 12, 10)

        self.lbl_titulo = QLabel(titulo)
        self.lbl_titulo.setProperty("titulo", True)
        self.lbl_valor = QLabel("R$ 0,00")
        self.lbl_valor.setProperty("valor", True)
        self.lbl_sub = QLabel(dica)
        self.lbl_sub.setProperty("sub", True)

        lay.addWidget(self.lbl_titulo)
        lay.addWidget(self.lbl_valor)
        lay.addWidget(self.lbl_sub)
        lay.addStretch(1)

    def set_value(self, v):
        self.lbl_valor.setText(fmt_money(v))

    def set_sub(self, s: str):
        self.lbl_sub.setText(s)


# ==============================
# Módulo 11 - Faturamento (reformulado: foco em Competência geral)
# ==============================
class ModuloFaturamento(QWidget):
    """
    Aba de Faturamento/DRE por COMPETÊNCIA (visão geral):
      - Filtro único: Competência (MM/AAAA)
      - Sem seletor de convênio ou unidade
      - KPIs consolidados a partir de tabelas mensais já estruturadas
    """
    
    def _norm_key(self, s: str) -> str:
        s = unicodedata.normalize("NFD", s)
        s = "".join(ch for ch in s if unicodedata.category(ch) != "Mn")  # remove acentos
        s = s.lower()
        s = re.sub(r"[\s_]+", "", s)     # remove espaços/underscores
        s = re.sub(r"[^\w]", "", s)      # remove pontuação
        return s

    def _find_col(self, con, table: str, *options: str):
        """
        Retorna o nome real da coluna existente em 'table', comparando de modo
        casefold + sem espaços + sem acentos + sem pontuação.
        """
        try:
            cur = con.cursor()
            cur.execute(f'PRAGMA table_info("{table}")')
            cols = [r[1] for r in cur.fetchall()]
            normcols = {self._norm_key(c): c for c in cols}
            for opt in options:
                key = self._norm_key(opt)
                if key in normcols:
                    return normcols[key]
        except Exception:
            pass
        return None
    
    def _find_col_tokens(self, con, table: str, *token_sets):
        """
        Procura uma coluna de 'table' que contenha TODOS os tokens de um dos 'token_sets'.
        Ex.: _find_col_tokens(con, "nfse_notas", ["lote","rps"]) encontra "Lote RPS" ou "RPS Lote".
        A comparação ignora acentos, espaços, underscores e pontuação.
        """
        try:
            cur = con.cursor()
            cur.execute(f'PRAGMA table_info("{table}")')
            cols = [r[1] for r in cur.fetchall()]
            # normaliza os nomes das colunas
            normcols = {c: self._norm_key(c) for c in cols}
            # para cada conjunto de tokens, tenta achar a 1ª coluna que contenha todos
            for tokens in token_sets:
                toks = [self._norm_key(t) for t in tokens]
                for col, nk in normcols.items():
                    if all(t in nk for t in toks):
                        return col
        except Exception:
            pass
        return None


    def _to_real(self, v):
        try:
            if v is None:
                return 0.0
            if isinstance(v, (int, float)):
                return float(v)
            s = str(v).strip()
            # remove qualquer coisa que não seja dígito, ponto, vírgula ou sinal
            s = re.sub(r"[^\d,.\-]", "", s)

            # tem vírgula e ponto? assume pt-BR: '.' milhar, ',' decimal
            if "," in s and "." in s:
                s = s.replace(".", "").replace(",", ".")
            elif "," in s:          # só vírgula → decimal
                s = s.replace(",", ".")
            # só dígitos → provável centavos
            if re.fullmatch(r"\d+", s):
                return float(s) / 100.0
            return float(s)
        except Exception:
            return 0.0

        
    def _mm_aaaa_from_date(self, txt: str) -> str:
        """Converte 'YYYY-MM-DD', 'DD/MM/YYYY' ou 'MM/YYYY' em 'MM/YYYY'. Retorna '' se não der pra entender."""
        if not txt:
            return ""
        s = str(txt).strip()
        # ISO: 2025-08-10
        m = re.match(r"^\s*(\d{4})-(\d{2})-(\d{2})", s)
        if m:
            return f"{m.group(2)}/{m.group(1)}"
        # BR: 10/08/2025
        m = re.search(r"(\d{1,2})/(\d{1,2})/(\d{4})", s)
        if m:
            return f"{int(m.group(2)):02d}/{m.group(3)}"
        # MM/AAAA
        m = re.search(r"(\d{1,2})/(\d{4})", s)
        if m:
            return f"{int(m.group(1)):02d}/{m.group(2)}"
        return ""

    def _safe_str(self, v):
        return "" if v is None else str(v)



    def __init__(self, parent=None):
        super().__init__(parent)
        self._montar_ui()
        self._popular_competencias()
        self._carregar_tudo()

    # -----------------------------
    # UI
    # -----------------------------
    def _montar_ui(self):
        root = QVBoxLayout(self)
        root.setContentsMargins(16, 16, 16, 16)
        root.setSpacing(12)

        # Topbar de filtros (apenas Competência)
        top = QHBoxLayout()
        top.setSpacing(8)

        top.addWidget(QLabel("Competência:"))
        self.combo_comp = QComboBox(); self.combo_comp.setMinimumWidth(110)
        top.addWidget(self.combo_comp)
        top.addStretch(1)

        self.btn_atualizar = QPushButton("Atualizar")
        self.btn_exportar = QPushButton("Exportar DRE")
        self.btn_bordero = QPushButton("Gerar Borderô")
        for b in (self.btn_atualizar, self.btn_exportar, self.btn_bordero):
            b.setStyleSheet(
                """
                QPushButton {background-color: #3498db; color: white; padding: 6px 12px; font-weight: bold; border-radius: 6px;}
                QPushButton:hover {background-color: #2980b9;}
                """
            )
        top.addWidget(self.btn_atualizar)
        top.addWidget(self.btn_exportar)
        top.addWidget(self.btn_bordero)

        root.addLayout(top)

        # Grid de KPIs (cards)
        grid = QGridLayout()
        grid.setHorizontalSpacing(12)
        grid.setVerticalSpacing(12)

        self.card_receita_bruta = KpiCard("Receita Bruta", "Soma dos registros faturados")
        self.card_deducoes = KpiCard("Deduções", "Canceladas/Substituídas/Descontos")
        self.card_receita_liquida = KpiCard("Receita Líquida")
        self.card_com_med = KpiCard("Comissões Médicas")
        self.card_com_vend = KpiCard("Comissões Vendedores")
        self.card_prest = KpiCard("Prestadores")
        self.card_func = KpiCard("Funcionários")
        self.card_impostos = KpiCard("Impostos")
        self.card_outros = KpiCard("Outros Custos")
        self.card_resultado_op = KpiCard("Resultado Operacional")
        self.card_lucro_liq = KpiCard("Lucro Líquido")

        cards = [
            self.card_receita_bruta, self.card_deducoes, self.card_receita_liquida,
            self.card_com_med, self.card_com_vend, self.card_prest,
            self.card_func, self.card_impostos, self.card_outros,
            self.card_resultado_op, self.card_lucro_liq
        ]

        # Dispor em 3 colunas
        colunas = 3
        for i, c in enumerate(cards):
            r = i // colunas
            k = i % colunas
            grid.addWidget(c, r, k)

        root.addLayout(grid)

        # Sub-abas
        self.tabs = QTabWidget()
        self.tab_visao = QWidget()
        self.tab_receitas = QWidget()
        self.tab_receber = QWidget()
        self.tab_pagar = QWidget()

        self.tabs.addTab(self.tab_visao, "Visão Geral")
        self.tabs.addTab(self.tab_receitas, "Receitas / NFSe")
        self.tabs.addTab(self.tab_receber, "Contas a Receber")
        self.tabs.addTab(self.tab_pagar, "Contas a Pagar")
        root.addWidget(self.tabs)

        # --- Visão Geral: tabela de resumo por competência (por convênio opcional no futuro) ---
        lay_v = QVBoxLayout(self.tab_visao)
        self.tbl_resumo = QTableWidget(0, 5)
        self.tbl_resumo.setHorizontalHeaderLabels([
            "Convênio", "Qtde Exames", "Receita Bruta", "Deduções", "Receita Líquida"
        ])
        self._style_table(self.tbl_resumo)
        lay_v.addWidget(self.tbl_resumo)

        # --- Receitas / NFSe ---
        # --- Receitas / NFSe ---
        lay_r = QVBoxLayout(self.tab_receitas)
        self.tbl_receitas = QTableWidget(0, 8)
        self.tbl_receitas.setHorizontalHeaderLabels([
            "Competência", "Convênio", "NFSe", "Lote", "Protocolo",
            "Situação", "Valor Nota", "Emitida em"
        ])
        self._style_table(self.tbl_receitas)
        lay_r.addWidget(self.tbl_receitas)



        # --- Contas a Receber ---
        lay_ar = QVBoxLayout(self.tab_receber)
        self.tbl_receber = QTableWidget(0, 6)
        self.tbl_receber.setHorizontalHeaderLabels([
            "Convênio", "Qtde", "Valor", "Previsto", "Recebido", "Atraso"
        ])
        self._style_table(self.tbl_receber)
        lay_ar.addWidget(self.tbl_receber)

        # --- Contas a Pagar ---
        lay_ap = QVBoxLayout(self.tab_pagar)
        self.tbl_pagar = QTableWidget(0, 6)
        self.tbl_pagar.setHorizontalHeaderLabels([
            "Centro de Custo", "Favorecido", "Competência", "Valor Calculado", "Valor Pago", "Status"
        ])
        self._style_table(self.tbl_pagar)
        lay_ap.addWidget(self.tbl_pagar)

        # Eventos
        self.combo_comp.currentIndexChanged.connect(self._carregar_tudo)
        self.btn_atualizar.clicked.connect(self._carregar_tudo)
        self.btn_exportar.clicked.connect(self._exportar_dre)
        self.btn_bordero.clicked.connect(self._gerar_bordero)

    def _style_table(self, tbl: QTableWidget):
        tbl.setEditTriggers(QAbstractItemView.NoEditTriggers)
        tbl.setSelectionBehavior(QAbstractItemView.SelectRows)
        tbl.verticalHeader().setVisible(False)
        tbl.horizontalHeader().setStretchLastSection(True)
        tbl.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

        # centraliza cabeçalho e células
        tbl.horizontalHeader().setDefaultAlignment(Qt.AlignCenter)
        tbl.setItemDelegate(CenterAlignDelegate(tbl))


    # -----------------------------
    # Filtros
    # -----------------------------
    def _popular_competencias(self):
        self.combo_comp.blockSignals(True)
        self.combo_comp.clear()
        try:
            con = sqlite3.connect(CAMINHO_BANCO)
            cur = con.cursor()
            # Ordena por ano/mês corretamente (MM/AAAA)
            cur.execute("""
                SELECT DISTINCT "Competência" AS comp
                FROM registros_financeiros
                WHERE TRIM("Competência") <> ''
            """)
            comps = [r[0] for r in cur.fetchall()]
            # ordena por AAAA/MM
            comps = sorted(comps, key=lambda s: (s[-4:], s[:2]))
            for c in comps:
                self.combo_comp.addItem(c)
            # seleciona a última competência disponível
            if comps:
                self.combo_comp.setCurrentText(comps[-1])
        except Exception as e:
            print("[Faturamento] Não foi possível carregar competências:", e)
        finally:
            try: con.close()
            except Exception: pass
        self.combo_comp.blockSignals(False)


    def _popular_convenios(self):
        # Lê da tabela 'convenios' se existir
        try:
            con = sqlite3.connect(CAMINHO_BANCO)
            cur = con.cursor()
            cur.execute("SELECT nome FROM convenios ORDER BY nome;")
            nomes = [r[0] for r in cur.fetchall()]
            for n in nomes:
                self.combo_convenio.addItem(n)
        except Exception:
            # Sem tabela/coluna - manter apenas "Todos"
            pass
        finally:
            try:
                con.close()
            except Exception:
                pass

    # -----------------------------
    # Carregamento geral
    # -----------------------------
    def _carregar_tudo(self):
        comp = self.combo_comp.currentText()
        # KPIs
        kpis = self._calcular_kpis(comp)
        self._atualizar_cards(kpis)
        # Tabelas
        self._carregar_resumo_por_convenio(comp)
        self._carregar_receitas(comp)
        self._carregar_a_receber(comp)
        self._carregar_a_pagar(comp)

    # -----------------------------
    # KPIs
    # -----------------------------
    def _calcular_kpis(self, competencia: str) -> dict:
        
        def _sum_outros_custos(con, comp):
            c = con.cursor()
            # 1) agregado
            c.execute("SELECT name FROM sqlite_master WHERE type='table' AND name='outros_custos_competencia'")
            if c.fetchone():
                c.execute("SELECT COALESCE(SUM(valor),0) FROM outros_custos_competencia WHERE competencia=?", (comp,))
                return safe_fetchone(c) or 0.0
            # 2) base
            c.execute("SELECT name FROM sqlite_master WHERE type='table' AND name='cadastro_outros_custos'")
            if c.fetchone():
                col_comp = self._find_col(con, "cadastro_outros_custos", "Competência", "competencia")
                col_val  = self._find_col(con, "cadastro_outros_custos", "Valor", "valor", "valor calculado", "valor_total", "valor_custo", "valor custo")
                if col_comp and col_val:
                    q = f"""
                        SELECT COALESCE(SUM(CAST(REPLACE(REPLACE({col_val},'.',''),',','.') AS REAL)),0)
                        FROM cadastro_outros_custos
                        WHERE {col_comp} = ?
                    """
                    c.execute(q, (comp,))
                    return safe_fetchone(c) or 0.0
            return 0.0

        
        rb = ded = com_med = com_vend = prest = func = imp = outros = 0.0
        try:
            con = sqlite3.connect(CAMINHO_BANCO)
            cur = con.cursor()

            # Receita Bruta (registros_financeiros)
            cur.execute('SELECT COALESCE(SUM("Valor Convenio"),0) FROM registros_financeiros WHERE "Competência" = ?;', (competencia,))
            rb = safe_fetchone(cur)

            # Deduções: se não houver tabela/coluna com valor de canceladas,
            # mantém 0. (No seu BD, nfse_notas tem "cancelada", mas sem valores.)
            ded = 0.0

            # Comissão Médica: pagamentos mensais, senão soma dos registros
            cur.execute('SELECT COALESCE(SUM(valor_pago),0) FROM pagamentos_medicos_competencia WHERE competencia=?;', (competencia,))
            com_med = safe_fetchone(cur)
            if not com_med:
                cur.execute('SELECT COALESCE(SUM("Valor Médico"),0) FROM registros_financeiros WHERE "Competência" = ?;', (competencia,))
                com_med = safe_fetchone(cur)

            # Comissão Vendedores: por convênio (vendedor_convenio + vendedores)
            cur.execute("""
                SELECT COALESCE(SUM( rf."Valor Convenio" *
                                    COALESCE(vc.comissao_especifica, v.comissao_padrao, 0) / 100.0), 0)
                FROM registros_financeiros rf
                LEFT JOIN vendedor_convenio vc ON vc.nome_convenio = rf.Convenio
                LEFT JOIN vendedores v ON v.id = vc.id_vendedor
                WHERE rf."Competência" = ?
            """, (competencia,))
            com_vend = safe_fetchone(cur)

            # Prestadores / Funcionários / Impostos / Outros Custos
            cur.execute('SELECT COALESCE(SUM(valor_pago),0) FROM pagamentos_prestadores WHERE competencia=?;', (competencia,))
            prest = safe_fetchone(cur)

            cur.execute('SELECT COALESCE(SUM(valor_pago),0) FROM pagamentos_funcionarios WHERE competencia=?;', (competencia,))
            func = safe_fetchone(cur)

            cur.execute('SELECT COALESCE(SUM(valor),0) FROM impostos_competencia WHERE competencia=?;', (competencia,))
            imp = safe_fetchone(cur)

            # valor em TEXT? Faz CAST seguro
            # depois (somando direto da base)
            outros = _sum_outros_custos(con, competencia)


        except Exception as e:
            print("[Faturamento] KPIs falharam:", e)
        finally:
            try: con.close()
            except Exception: pass

        rl = max(rb - ded, 0.0)
        resultado_op = rl - (com_med + com_vend + prest + func + imp + outros)
        lucro_liq = resultado_op

        return {
            'receita_bruta': rb,
            'deducoes': ded,
            'receita_liquida': rl,
            'com_med': com_med,
            'com_vend': com_vend,
            'prest': prest,
            'func': func,
            'imp': imp,
            'outros': outros,
            'resultado_op': resultado_op,
            'lucro_liq': lucro_liq,
        }


    def _atualizar_cards(self, kpis: dict):
        self.card_receita_bruta.set_value(kpis.get('receita_bruta', 0.0))
        self.card_deducoes.set_value(kpis.get('deducoes', 0.0))
        self.card_receita_liquida.set_value(kpis.get('receita_liquida', 0.0))
        self.card_com_med.set_value(kpis.get('com_med', 0.0))
        self.card_com_vend.set_value(kpis.get('com_vend', 0.0))
        self.card_prest.set_value(kpis.get('prest', 0.0))
        self.card_func.set_value(kpis.get('func', 0.0))
        self.card_impostos.set_value(kpis.get('imp', 0.0))
        self.card_outros.set_value(kpis.get('outros', 0.0))
        self.card_resultado_op.set_value(kpis.get('resultado_op', 0.0))
        self.card_lucro_liq.set_value(kpis.get('lucro_liq', 0.0))
# -----------------------------
    # Tabelas
    # -----------------------------
    def _carregar_resumo_por_convenio(self, competencia: str):
        self.tbl_resumo.setRowCount(0)
        rows = []
        try:
            con = sqlite3.connect(CAMINHO_BANCO)
            cur = con.cursor()
            cur.execute("""
                SELECT Convenio, COUNT(*) AS qt, COALESCE(SUM("Valor Convenio"),0) AS soma
                FROM registros_financeiros
                WHERE "Competência" = ?
                GROUP BY Convenio
                ORDER BY Convenio
            """, (competencia,))
            rows = cur.fetchall()
        except Exception as e:
            print("[Faturamento] Resumo por convênio falhou:", e)
        finally:
            try: con.close()
            except Exception: pass

        for conv, qtde, soma in rows:
            r = self.tbl_resumo.rowCount()
            self.tbl_resumo.insertRow(r)
            self.tbl_resumo.setItem(r, 0, QTableWidgetItem(str(conv)))
            self.tbl_resumo.setItem(r, 1, QTableWidgetItem(str(qtde)))
            self.tbl_resumo.setItem(r, 2, QTableWidgetItem(fmt_money(soma)))
            self.tbl_resumo.setItem(r, 3, QTableWidgetItem(fmt_money(0)))  # Deduções (0 por ora)
            self.tbl_resumo.setItem(r, 4, QTableWidgetItem(fmt_money(max(soma - 0, 0.0))))


    def _carregar_receitas(self, competencia: str):
        """1 linha por convênio com NFSe, Lote, Protocolo, Situação, Valor Nota e Data de Emissão."""
        self.tbl_receitas.setRowCount(0)
        con = None
        try:
            con = sqlite3.connect(CAMINHO_BANCO)
            cur = con.cursor()

            # ---- Convênios que têm movimento na competência (base da grade) ----
            cur.execute("""
                SELECT DISTINCT Convenio
                FROM registros_financeiros
                WHERE "Competência" = ? AND TRIM(Convenio) <> ''
            """, (competencia,))
            convenios = [self._safe_str(r[0]) for r in cur.fetchall()]

            # ---- Soma dos RPS por protocolo (valor da nota) ----
            # ---- Soma do valor da nota por PROTOCOLO (nfse_rps) ----
            rps_sum = {}
            try:
                cur.execute("""SELECT protocolo, valor_servicos FROM nfse_rps""")
                for protocolo, valor_txt in cur.fetchall():
                    p = self._safe_str(protocolo)
                    v = self._to_real(valor_txt)   # trata "336,50", "336.50" ou "33650"
                    rps_sum[p] = rps_sum.get(p, 0.0) + v
            except Exception as e:
                print("[NFSe] Falha ao somar nfse_rps:", e)


            def ultimo_lote(conv):
                cur.execute("""
                    SELECT protocolo, numero_lote
                    FROM nfse_lotes
                    WHERE convenio_nome = ? AND competencia = ?
                    ORDER BY id DESC LIMIT 1
                """, (conv, competencia))
                row = cur.fetchone()
                if row:
                    return self._safe_str(row[0]), self._safe_str(row[1])
                return "", ""

            def numero_nf(prot, conv):
                # preferencial: pela relação protocolo -> nfse_notas
                if prot:
                    cur.execute("""
                        SELECT numero_nfse
                        FROM nfse_notas
                        WHERE protocolo = ?
                        ORDER BY id DESC LIMIT 1
                    """, (prot,))
                    r = cur.fetchone()
                    if r and r[0]:
                        return self._safe_str(r[0])
                # fallback (caso tenha mais de um lote no mês)
                cur.execute("""
                    SELECT n.numero_nfse
                    FROM nfse_notas n
                    JOIN nfse_lotes l ON l.protocolo = n.protocolo
                    WHERE l.convenio_nome = ? AND l.competencia = ?
                    ORDER BY n.id DESC
                    LIMIT 1
                """, (conv, competencia))
                r = cur.fetchone()
                return self._safe_str(r[0]) if r else ""

            def status_e_emissao(conv):
                # tenta por competencia_servico, competencia (texto) ou competencia_nfse
                cur.execute("""
                    SELECT status, data_emissao
                    FROM notas_emitidas_status
                    WHERE convenio = ?
                    AND (competencia_servico = ? OR competencia = ? OR competencia_nfse = ?)
                    ORDER BY id DESC LIMIT 1
                """, (conv, competencia, competencia, competencia))
                r = cur.fetchone()
                st = self._safe_str(r[0]) if r and len(r) > 0 else "Pendente"
                de = self._safe_str(r[1]) if r and len(r) > 1 else ""
                return st, de

            linhas = []
            for conv in sorted(set(convenios)):
                prot, num_lote = ultimo_lote(conv)
                nf   = numero_nf(prot, conv)
                st, de = status_e_emissao(conv)

                # valor: se tiver um protocolo "principal", usa; senão soma todos os protocolos do mês
                if prot:
                    valor = rps_sum.get(prot, 0.0)
                else:
                    cur.execute("""
                        SELECT protocolo FROM nfse_lotes
                        WHERE convenio_nome = ? AND competencia = ?
                    """, (conv, competencia))
                    prot_list = [self._safe_str(p[0]) for p in cur.fetchall() if p and p[0]]
                    valor = sum(rps_sum.get(p, 0.0) for p in prot_list)

                linhas.append({
                    "comp": competencia,
                    "conv": conv,
                    "nfse": nf or "-",
                    "lote": num_lote or "-",
                    "prot": prot or "-",
                    "status": st or ("Emitida" if nf else "Pendente"),
                    "valor": float(valor),
                    "emissao": de,
                })

            # ---- Renderiza ----
            for d in linhas:
                r = self.tbl_receitas.rowCount()
                self.tbl_receitas.insertRow(r)
                valores = [
                    d["comp"], d["conv"], d["nfse"], d["lote"], d["prot"],
                    d["status"], d["valor"], d["emissao"]
                ]
                for c, val in enumerate(valores):
                    if c == 6:  # Valor Nota
                        item = QTableWidgetItem(fmt_money(val))
                        item.setTextAlignment(Qt.AlignCenter)

                    else:
                        item = QTableWidgetItem(str(val))
                    self.tbl_receitas.setItem(r, c, item)

        except Exception as e:
            print("[Faturamento] NFSe (por convênio) falhou:", e)
        finally:
            try:
                if con: con.close()
            except Exception:
                pass








    def _carregar_a_receber(self, competencia: str):
        self.tbl_receber.setRowCount(0)
        try:
            con = sqlite3.connect(CAMINHO_BANCO)
            cur = con.cursor()
            cur.execute("""
                SELECT Convenio, COUNT(*) AS qt, COALESCE(SUM("Valor Convenio"),0) AS soma
                FROM registros_financeiros
                WHERE "Competência" = ?
                GROUP BY Convenio
                ORDER BY Convenio
            """, (competencia,))
            rows = cur.fetchall()
        except Exception as e:
            print("[Faturamento] A Receber falhou:", e)
            rows = []
        finally:
            try: con.close()
            except Exception: pass

        for conv, qtde, soma in rows:
            r = self.tbl_receber.rowCount()
            self.tbl_receber.insertRow(r)
            self.tbl_receber.setItem(r, 0, QTableWidgetItem(str(conv)))
            self.tbl_receber.setItem(r, 1, QTableWidgetItem(str(qtde)))
            self.tbl_receber.setItem(r, 2, QTableWidgetItem(fmt_money(soma)))
            self.tbl_receber.setItem(r, 3, QTableWidgetItem("-"))           # Previsto (quando tiver tabela própria)
            self.tbl_receber.setItem(r, 4, QTableWidgetItem(fmt_money(0)))   # Recebido (conciliação futura)
            self.tbl_receber.setItem(r, 5, QTableWidgetItem("-"))            # Atraso/aging


    def _carregar_a_pagar(self, competencia: str):
        self.tbl_pagar.setRowCount(0)
        k = self._calcular_kpis(competencia)
        centros = [
            ("Médicos", "-", competencia, k.get("com_med", 0.0), 0.0, "Previsto"),
            ("Vendedores", "-", competencia, k.get("com_vend", 0.0), 0.0, "Previsto"),
            ("Prestadores", "-", competencia, k.get("prest", 0.0), 0.0, "Previsto"),
            ("Funcionários", "-", competencia, k.get("func", 0.0), 0.0, "Previsto"),
            ("Impostos", "-", competencia, k.get("imp", 0.0), 0.0, "Previsto"),
            ("Outros Custos", "-", competencia, k.get("outros", 0.0), 0.0, "Previsto"),
        ]
        for row in centros:
            r = self.tbl_pagar.rowCount()
            self.tbl_pagar.insertRow(r)
            for c, val in enumerate(row):
                if c in (3, 4):
                    item = QTableWidgetItem(fmt_money(val))
                    item.setTextAlignment(Qt.AlignCenter)

                else:
                    item = QTableWidgetItem(str(val))
                self.tbl_pagar.setItem(r, c, item)


    # -----------------------------
    # Exportações
    # -----------------------------
    def _exportar_dre(self):
        """Exporta os KPIs agregados do topo para CSV simples."""
        kpis = {
            "Receita Bruta": self.card_receita_bruta.lbl_valor.text(),
            "Deduções": self.card_deducoes.lbl_valor.text(),
            "Receita Líquida": self.card_receita_liquida.lbl_valor.text(),
            "Comissões Médicas": self.card_com_med.lbl_valor.text(),
            "Comissões Vendedores": self.card_com_vend.lbl_valor.text(),
            "Prestadores": self.card_prest.lbl_valor.text(),
            "Funcionários": self.card_func.lbl_valor.text(),
            "Impostos": self.card_impostos.lbl_valor.text(),
            "Outros Custos": self.card_outros.lbl_valor.text(),
            "Resultado Operacional": self.card_resultado_op.lbl_valor.text(),
            "Lucro Líquido": self.card_lucro_liq.lbl_valor.text(),
        }
        path, _ = QFileDialog.getSaveFileName(self, "Salvar DRE (CSV)", "dre.csv", "CSV (*.csv)")
        if not path:
            return
        try:
            with open(path, "w", encoding="utf-8") as f:
                f.write("KPI;Valor\n")
                for k, v in kpis.items():
                    f.write(f"{k};{v}\n")

            QMessageBox.information(self, "Exportação", "DRE exportado com sucesso!")
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Falha ao exportar: {e}")

    def _gerar_bordero(self):
        """Gera um CSV simples com as linhas da aba 'Contas a Pagar'."""
        path, _ = QFileDialog.getSaveFileName(self, "Salvar Borderô (CSV)", "bordero.csv", "CSV (*.csv)")
        if not path:
            return
        try:
            with open(path, "w", encoding="utf-8") as f:
                f.write("Centro de Custo;Favorecido;Competência;Valor Calculado;Valor Pago;Status\n")

                for r in range(self.tbl_pagar.rowCount()):
                    vals = []
                    for c in range(self.tbl_pagar.columnCount()):
                        item = self.tbl_pagar.item(r, c)
                        vals.append((item.text() if item else "").replace(";", ","))
                    f.write(";".join(vals) + "\n")

            QMessageBox.information(self, "Borderô", "Arquivo gerado com sucesso!")
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Falha ao gerar borderô: {e}")

