# gui/modulo14_vendas.py
# -*- coding: utf-8 -*-
from __future__ import annotations
import os
import re
import json
import sqlite3
import unicodedata
from datetime import datetime, timedelta
from typing import List, Dict, Optional, Tuple

from PyQt5.QtCore import Qt, QDate, QDateTime, QSize
from PyQt5.QtWidgets import (
    QWidget, QTabWidget, QVBoxLayout, QHBoxLayout, QLabel, QPushButton,
    QLineEdit, QComboBox, QDateEdit, QDateTimeEdit, QTextEdit,
    QTableWidget, QTableWidgetItem, QHeaderView, QAbstractItemView,
    QListWidget, QListWidgetItem, QFrame, QSplitter, QScrollArea,
    QFileDialog, QMessageBox, QMenu, QAction, QSpinBox, QDoubleSpinBox, 
)
from PyQt5.QtWidgets import QMenu, QAction, QSpinBox, QDoubleSpinBox, QDialog, QDialogButtonBox

# Caminho do banco (mesmo do projeto)
CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

# ------------------------------ SCHEMA ------------------------------

def ensure_schema_vendas():
    """Cria (se não existirem) as tabelas do módulo de Vendas."""
    con = sqlite3.connect(CAMINHO_BANCO)
    cur = con.cursor()

    cur.execute(
        """
        CREATE TABLE IF NOT EXISTS leads (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            empresa_raw   TEXT,
            empresa_norm  TEXT,
            contato_nome  TEXT,
            contato_email TEXT,
            contato_tel   TEXT,
            cidade        TEXT,
            uf            TEXT,
            origem        TEXT,
            lista         TEXT CHECK(lista IN ('FRIA','MORNA','QUENTE')) DEFAULT 'FRIA',
            score         REAL DEFAULT 0,
            tags          TEXT,
            observacoes   TEXT,
            dt_criado     TEXT,
            dt_atualizado TEXT
        )
        """
    )

    cur.execute(
        """
        CREATE TABLE IF NOT EXISTS oportunidades (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            lead_id        INTEGER REFERENCES leads(id) ON DELETE CASCADE,
            titulo         TEXT,
            valor_estimado REAL DEFAULT 0,
            prob           REAL DEFAULT 0.5,
            estagio        TEXT CHECK(estagio IN (
                               'LEAD','QUALIFICADO','PROPOSTA','NEGOCIACAO','FECHADO_GANHO','FECHADO_PERDIDO'
                           )) DEFAULT 'LEAD',
            perdido_motivo TEXT,
            dt_criado      TEXT,
            dt_fechado     TEXT,
            dt_atualizado  TEXT
        )
        """
    )

    cur.execute(
        """
        CREATE TABLE IF NOT EXISTS atividades (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            lead_id         INTEGER REFERENCES leads(id) ON DELETE CASCADE,
            oportunidade_id INTEGER REFERENCES oportunidades(id) ON DELETE CASCADE,
            tipo            TEXT CHECK(tipo IN ('TAREFA','EMAIL','WHATSAPP','LIGACAO','REUNIAO','ANOTACAO')) DEFAULT 'TAREFA',
            descricao       TEXT,
            status_tarefa   TEXT CHECK(status_tarefa IN ('PENDENTE','FEITA','CANCELADA')) DEFAULT 'PENDENTE',
            dt_prevista     TEXT,
            dt_realizada    TEXT,
            responsavel     TEXT,
            dt_criado       TEXT
        )
        """
    )

    cur.execute(
        """
        CREATE TABLE IF NOT EXISTS cadencias (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            nome        TEXT,
            passos_json TEXT,
            ativo       INTEGER DEFAULT 1
        )
        """
    )
    
    cur.execute(
    """
        CREATE TABLE IF NOT EXISTS itens_oportunidade (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            oportunidade_id INTEGER REFERENCES oportunidades(id) ON DELETE CASCADE,
            servico       TEXT,
            quantidade    INTEGER DEFAULT 1,
            valor_unit    REAL DEFAULT 0
        )
        """
    )
    cur.execute("CREATE INDEX IF NOT EXISTS idx_itens_opp ON itens_oportunidade(oportunidade_id)")


    # Índices úteis
    cur.execute("CREATE INDEX IF NOT EXISTS idx_leads_norm   ON leads(empresa_norm)")
    cur.execute("CREATE INDEX IF NOT EXISTS idx_opps_stage   ON oportunidades(estagio)")
    cur.execute("CREATE INDEX IF NOT EXISTS idx_ativ_status  ON atividades(status_tarefa)")
    cur.execute("CREATE INDEX IF NOT EXISTS idx_ativ_dtprev  ON atividades(dt_prevista)")

    con.commit()

    # Se não houver nenhuma cadência, cria uma padrão
    cur.execute("SELECT COUNT(*) FROM cadencias")
    n = cur.fetchone()[0]
    if n == 0:
        passos = [
            {"dias": 0, "tipo": "EMAIL",    "descricao": "Apresentação e proposta de valor"},
            {"dias": 2, "tipo": "WHATSAPP", "descricao": "Follow-up curto perguntando sobre disponibilidade"},
            {"dias": 5, "tipo": "LIGACAO",  "descricao": "Ligação para alinhar necessidade e TAT"},
        ]
        cur.execute(
            "INSERT INTO cadencias(nome, passos_json, ativo) VALUES (?,?,1)",
            ("Cadência Padrão (7 dias)", json.dumps(passos, ensure_ascii=False))
        )
        con.commit()

    con.close()


# ------------------------------ UTILS ------------------------------

def _now_str() -> str:
    return datetime.now().strftime("%Y-%m-%d %H:%M:%S")


def _normalize_empresa(s: str) -> str:
    """Normaliza nome de empresa (sem acento, sem sufixos, caixa alta, sem ruído)."""
    t = str(s or "").strip().upper()
    t = unicodedata.normalize('NFKD', t)
    t = ''.join(ch for ch in t if not unicodedata.combining(ch))
    t = re.sub(r'[^A-Z0-9 ]+', ' ', t)
    t = re.sub(r'\s+', ' ', t).strip()
    STOP = {
        'LTDA','EIRELI','ME','EPP','SA','S A','HOLDING','GRUPO',
        'HOSPITAL','CLINICA','CLINICO','LABORATORIO','LAB','SERVICOS','SERVICO',
        'ASSOCIACAO','COOPERATIVA','FUNDACAO','INSTITUTO',
        'DE','DA','DO','DAS','DOS','SAUDE','SISTEMA'
    }
    tokens = [tok for tok in t.split() if tok not in STOP and len(tok) > 1]
    if not tokens:
        tokens = [w for w in t.split() if len(w) > 1] or [t]
    return ' '.join(tokens)


# ------------------------------ KANBAN ------------------------------

ESTAGIOS = ['LEAD','QUALIFICADO','PROPOSTA','NEGOCIACAO','FECHADO_GANHO','FECHADO_PERDIDO']
ESTAGIO_LABEL = {
    'LEAD': 'Lead',
    'QUALIFICADO': 'Qualificado',
    'PROPOSTA': 'Proposta',
    'NEGOCIACAO': 'Negociação',
    'FECHADO_GANHO': 'Ganho',
    'FECHADO_PERDIDO': 'Perdido',
}

class KanbanList(QListWidget):
    def __init__(self, parent: 'ModuloVendas', stage: str):
        super().__init__()
        self.parent_mod = parent
        self.stage = stage
        self.setAcceptDrops(True)
        self.setDragEnabled(True)
        self.setDefaultDropAction(Qt.MoveAction)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        self.setStyleSheet("QListWidget{background:#f8fafc;border:1px solid #e5e7eb;border-radius:8px}")
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self._open_menu)
        self.itemDoubleClicked.connect(lambda it: self.parent_mod._editar_oportunidade_from_item(it))
        
    def _open_menu(self, pos):
        item = self.itemAt(pos)
        if not item: return
        opp_id = item.data(Qt.UserRole)
        menu = QMenu(self)
        act_open = QAction("Abrir/Editar", self); act_del = QAction("Excluir oportunidade", self)
        act_open.triggered.connect(lambda: self.parent_mod._editar_oportunidade(opp_id))
        act_del.triggered.connect(lambda: self.parent_mod._excluir_oportunidade(opp_id))
        menu.addAction(act_open); menu.addAction(act_del)
        menu.exec_(self.viewport().mapToGlobal(pos))



    def dragEnterEvent(self, e): e.accept()
    def dragMoveEvent(self, e): e.accept()

    def dropEvent(self, e):
        super().dropEvent(e)
        # após o drop, descobrir o item e atualizar o estágio no banco
        item = self.currentItem()
        if not item: return
        opp_id = item.data(Qt.UserRole)
        self.parent_mod._mover_oportunidade(opp_id, self.stage)


# ------------------------------ MÓDULO VENDAS ------------------------------

class ModuloVendas(QTabWidget):
    """Módulo 14 – Vendas (Leads, Pipeline, Tarefas, Cadências, Mensagens, Relatórios)."""
    
    def _short(self, s: str, n: int = 90) -> str:
        s = str(s or "").strip()
        return (s[:n] + "…") if len(s) > n else s

    
    def _fetch_itens_por_opp(self) -> dict:
        """Retorna {oportunidade_id: [(servico, valor_unit, quantidade)]}."""
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("""
            SELECT oportunidade_id, COALESCE(servico,''), COALESCE(valor_unit,0), COALESCE(quantidade,1)
            FROM itens_oportunidade
            ORDER BY oportunidade_id, id
        """)
        mapa = {}
        for opp_id, srv, vu, qtd in cur.fetchall():
            mapa.setdefault(int(opp_id), []).append((str(srv), float(vu or 0.0), int(qtd or 1)))
        con.close()
        return mapa

    def _fmt_money(self, v: float) -> str:
        s = f"{float(v):,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        return f"R$ {s}"

    
    def _editar_oportunidade_from_item(self, item):
        if not item: return
        self._editar_oportunidade(int(item.data(Qt.UserRole)))
        
    def _refresh_leads_combos(self):
        """Recarrega combos de leads nas abas Tarefas, Cadências e Mensagens."""
        leads = self._listar_leads()
        self._last_new_lead_id = getattr(self, "_last_new_lead_id", None)

        def refill(cb):
            if cb is None: return
            current = cb.currentData()
            cb.blockSignals(True)
            cb.clear()
            for lid, nome in leads:
                cb.addItem(nome, lid)
            target = self._last_new_lead_id if self._last_new_lead_id else current
            if target is not None:
                idx = cb.findData(target)
                if idx >= 0: cb.setCurrentIndex(idx)
            cb.blockSignals(False)

        refill(getattr(self, "cb_task_lead", None))
        refill(getattr(self, "cb_cad_lead", None))
        refill(getattr(self, "cb_msg_lead", None))




    def _editar_oportunidade(self, opp_id: int):
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("SELECT lead_id, titulo, prob, estagio, perdido_motivo FROM oportunidades WHERE id=?", (opp_id,))
        r = cur.fetchone()
        if not r:
            con.close(); return
        lid, tit, prob, est, motivo = r

        # carrega itens existentes
        cur.execute("SELECT id, servico, COALESCE(valor_unit,0) FROM itens_oportunidade WHERE oportunidade_id=? ORDER BY id", (opp_id,))
        itens = cur.fetchall()
        con.close()

        dlg = QDialog(self); dlg.setWindowTitle("Editar Oportunidade")
        v = QVBoxLayout(dlg)

        # topo: Lead (somente leitura), Título, Prob, Estágio, Motivo (se perdido)
        top = QHBoxLayout()
        lbl_lead = QLabel(next((n for (i,n) in self._listar_leads() if i==lid), "Lead"))
        ed_tit = QLineEdit(tit or "Oportunidade")
        prob_cb = QComboBox(); prob_cb.addItems(["0.2","0.3","0.5","0.7","0.9"]); prob_cb.setCurrentText(f"{float(prob or 0.5):.1f}")
        est_cb  = QComboBox(); est_cb.addItems(ESTAGIOS); est_cb.setCurrentText(est or "LEAD")
        ed_mot  = QLineEdit(motivo or "")
        for w in (QLabel("Lead:"), lbl_lead, QLabel("Título:"), ed_tit, QLabel("Prob:"), prob_cb, QLabel("Estágio:"), est_cb, QLabel("Motivo:"), ed_mot):
            top.addWidget(w)
        top.addStretch(1)
        v.addLayout(top)

        # tabela de serviços: Serviço | Valor (R$)
        tbl = QTableWidget(); tbl.setColumnCount(3)
        tbl.setHorizontalHeaderLabels(["Serviço","Valor (R$)","_id"])
        tbl.hideColumn(2)
        tbl.verticalHeader().setVisible(False)
        tbl.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        v.addWidget(tbl)

        def add_row(srv="ECG", vu=0.0, iid=None):
            r = tbl.rowCount(); tbl.insertRow(r)
            tbl.setItem(r, 0, QTableWidgetItem(str(srv or "")))
            spv = QDoubleSpinBox(); spv.setRange(0, 1_000_000); spv.setDecimals(2); spv.setPrefix("R$ "); spv.setValue(float(vu or 0.0))
            tbl.setCellWidget(r, 1, spv)
            tbl.setItem(r, 2, QTableWidgetItem("" if iid is None else str(int(iid))))
        for iid, srv, vu in itens:
            add_row(srv, vu, iid)

        hb = QHBoxLayout()
        btn_add = QPushButton("Adicionar serviço"); btn_del = QPushButton("Excluir serviço selecionado")
        hb.addWidget(btn_add); hb.addWidget(btn_del); hb.addStretch(1)
        v.addLayout(hb)
        btn_add.clicked.connect(lambda: add_row())

        def del_row():
            r = tbl.currentRow()
            if r < 0: return
            iid_txt = tbl.item(r, 2).text() if tbl.item(r, 2) else ""
            if iid_txt:
                con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
                cur.execute("DELETE FROM itens_oportunidade WHERE id=?", (int(iid_txt),))
                con.commit(); con.close()
            tbl.removeRow(r)
        btn_del.clicked.connect(del_row)

        bb = QDialogButtonBox(QDialogButtonBox.Save | QDialogButtonBox.Cancel)
        v.addWidget(bb)

        def on_save():
            con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
            cur.execute("""UPDATE oportunidades SET titulo=?, prob=?, estagio=?, perdido_motivo=?, dt_atualizado=? WHERE id=?""",
                        (ed_tit.text().strip() or "Oportunidade", float(prob_cb.currentText()), est_cb.currentText(), ed_mot.text().strip(), _now_str(), opp_id))
            # upserts dos itens
            rows = tbl.rowCount()
            for r in range(rows):
                srv = (tbl.item(r, 0).text() if tbl.item(r, 0) else "").strip() or "Serviço"
                vu  = tbl.cellWidget(r, 1).value() if tbl.cellWidget(r, 1) else 0.0
                iid_txt = tbl.item(r, 2).text() if tbl.item(r, 2) else ""
                if iid_txt:
                    cur.execute("UPDATE itens_oportunidade SET servico=?, valor_unit=?, quantidade=1 WHERE id=?", (srv, float(vu), int(iid_txt)))
                else:
                    cur.execute("INSERT INTO itens_oportunidade(oportunidade_id, servico, quantidade, valor_unit) VALUES (?,?,?,?)",
                                (opp_id, srv, 1, float(vu)))
            con.commit(); con.close()

            self._load_pipeline()
            self._calc_kpis()
            dlg.accept()

        bb.accepted.connect(on_save)
        bb.rejected.connect(dlg.reject)
        dlg.exec_()


    def _excluir_oportunidade(self, opp_id: int):
        if QMessageBox.question(self, "Excluir Oportunidade", "Confirmar exclusão desta oportunidade? Tarefas vinculadas serão removidas.") != QMessageBox.Yes:
            return
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("DELETE FROM oportunidades WHERE id=?", (opp_id,))
        con.commit(); con.close()
        self._load_pipeline()

    
    def _importar_csv(self):
        """Importa leads a partir de um CSV. Tenta ; ou , e UTF-8/Latin-1."""
        path, _ = QFileDialog.getOpenFileName(self, "Importar leads (CSV)", "", "CSV (*.csv)")
        if not path:
            return

        import csv

        # Lê o arquivo tentando detectar delimitador e encoding
        rows = None
        for enc in ("utf-8-sig", "latin1"):
            try:
                with open(path, "r", encoding=enc, newline="") as f:
                    sample = f.read(4096)
                    f.seek(0)
                    try:
                        dialect = csv.Sniffer().sniff(sample, delimiters=";,")
                    except Exception:
                        # fallback: tenta ; se existir no sample, senão ,
                        dialect = csv.excel
                        dialect.delimiter = ";" if ";" in sample and sample.count(";") >= sample.count(",") else ","
                    reader = csv.DictReader(f, dialect=dialect)
                    rows = list(reader)
                break
            except Exception:
                rows = None

        if not rows:
            QMessageBox.warning(self, "Importar CSV", "Não foi possível ler o arquivo CSV.")
            return

        def pick(d: dict, keys: list) -> str:
            """Pega um campo do dict por nome (case-insensitive)."""
            low = { (k or "").strip().lower(): (d[k] or "") for k in d.keys() }
            for k in keys:
                k = k.lower()
                if k in low:
                    return str(low[k]).strip()
            return ""

        con = sqlite3.connect(CAMINHO_BANCO)
        cur = con.cursor()

        ok, skip = 0, 0
        for r in rows:
            empresa = pick(r, ["empresa", "empresa_raw", "nome", "razão social", "razao social"])
            if not empresa:
                skip += 1
                continue
            contato = pick(r, ["contato", "responsável", "responsavel", "nome_contato"])
            email   = pick(r, ["email", "e-mail"])
            tel     = pick(r, ["telefone", "tel", "celular", "whatsapp"])
            cidade  = pick(r, ["cidade", "município", "municipio"])
            uf_val  = pick(r, ["uf", "estado", "sigla_uf"])
            uf      = uf_val.upper()[:2] if uf_val else None
            origem  = pick(r, ["origem"]) or "CSV"
            lista   = pick(r, ["lista", "temperatura"]).upper()
            if lista not in ("FRIA", "MORNA", "QUENTE"):
                lista = "FRIA"
            tags    = pick(r, ["tags", "marcadores"])

            score_s = pick(r, ["score", "pontuação", "pontuacao"])
            try:
                score = float(score_s.replace(",", ".")) if score_s else 0.0
            except Exception:
                score = 0.0

            cur.execute(
                """
                INSERT INTO leads(
                    empresa_raw, empresa_norm, contato_nome, contato_email, contato_tel,
                    cidade, uf, origem, lista, score, tags, dt_criado, dt_atualizado
                ) VALUES (?,?,?,?,?,?,?,?,?,?,?, ?, ?)
                """,
                (
                    empresa, _normalize_empresa(empresa), contato, email, tel,
                    cidade, uf, origem, lista, score, tags, _now_str(), _now_str()
                )
            )
            ok += 1

        con.commit()
        con.close()

        self._load_leads()
        QMessageBox.information(
            self, "Importar CSV",
            f"Importados: {ok}. Ignorados (sem nome de empresa): {skip}."
        )


    def __init__(self, parent=None):
        super().__init__(parent)
        ensure_schema_vendas()

        # Abas
        self.tab_pipeline = QWidget(); self.addTab(self.tab_pipeline, "Pipeline")
        self.tab_leads    = QWidget(); self.addTab(self.tab_leads,    "Leads")
        self.tab_tasks    = QWidget(); self.addTab(self.tab_tasks,    "Tarefas")
        self.tab_cad      = QWidget(); self.addTab(self.tab_cad,      "Cadências")
        self.tab_msg      = QWidget(); self.addTab(self.tab_msg,      "Mensagens")
        self.tab_rep      = QWidget(); self.addTab(self.tab_rep,      "Relatórios")

        self._montar_pipeline()
        self._montar_leads()
        self._montar_tarefas()
        self._montar_cadencias()
        self._montar_mensagens()
        self._montar_relatorios()

        self._apply_style()

    # ----------------- PIPELINE -----------------
    def _montar_pipeline(self):
        root = QVBoxLayout(self.tab_pipeline); root.setContentsMargins(12,12,12,12); root.setSpacing(8)

        top = QHBoxLayout()
        self.btn_refresh_pipe = QPushButton("Atualizar"); self.btn_refresh_pipe.clicked.connect(self._load_pipeline)
        self.btn_new_opp = QPushButton("Nova Oportunidade…"); self.btn_new_opp.clicked.connect(self._nova_oportunidade)
        top.addWidget(self.btn_refresh_pipe); top.addSpacing(12); top.addWidget(self.btn_new_opp); top.addStretch(1)
        root.addLayout(top)

        # header com nomes
        self._pipe_headers = {}
        header = QHBoxLayout()
        for st in ESTAGIOS:
            lbl = QLabel(f"{ESTAGIO_LABEL[st]} (0)")
            lbl.setStyleSheet("font-weight:700;color:#334155")
            self._pipe_headers[st] = lbl
            header.addWidget(lbl)
        root.addLayout(header)

        # área rolável contendo as colunas Kanban
        self.scroll = QScrollArea(); self.scroll.setWidgetResizable(True)
        host = QWidget(); self.scroll.setWidget(host)
        lay = QHBoxLayout(host); lay.setContentsMargins(0,0,0,0); lay.setSpacing(8)

        self.kanban_cols: Dict[str, KanbanList] = {}
        for st in ESTAGIOS:
            col = KanbanList(self, st); col.setMinimumWidth(220)
            self.kanban_cols[st] = col
            lay.addWidget(col)
        root.addWidget(self.scroll, 1)

        self._load_pipeline()

    def _load_pipeline(self):
        # limpa colunas
        for st, lw in self.kanban_cols.items():
            lw.clear()

        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("""
            SELECT o.id,
                o.titulo,
                COALESCE( (SELECT SUM(COALESCE(i.quantidade,1)*COALESCE(i.valor_unit,0))
                            FROM itens_oportunidade i WHERE i.oportunidade_id = o.id),
                            o.valor_estimado, 0) AS valor_total,
                o.prob, o.estagio,
                COALESCE(l.empresa_raw, l.empresa_norm, '') AS lead_nome,
                COALESCE(l.cidade, ''), COALESCE(l.uf, ''),
                COALESCE((SELECT MIN(dt_prevista) FROM atividades a WHERE a.oportunidade_id=o.id AND a.status_tarefa='PENDENTE'), '') as proxima,
                COALESCE(o.perdido_motivo, '') AS motivo
            FROM oportunidades o
            LEFT JOIN leads l ON l.id=o.lead_id
            ORDER BY o.dt_atualizado DESC
        """)

        rows = cur.fetchall()
        con.close()

        # pega itens de todas as opps de uma vez
        itens_map = self._fetch_itens_por_opp()

        for opp_id, titulo, valor_total, prob, estagio, lead_nome, cidade, uf, prox, motivo in rows:
            # monta linhas do card
            linhas = []
            linhas.append(f"{lead_nome}")
            if cidade or uf:
                linhas[-1] += f"  •  {cidade}-{uf}".strip(" -")
            linhas.append(titulo or "Oportunidade")

            itens = itens_map.get(int(opp_id), [])
            if itens:
                for srv, vu, qtd in itens:
                    # exibimos só o valor; se houver qtd>1, somamos ao total mas não mostramos a qtd (pedido do usuário)
                    linhas.append(f"• {srv} — {self._fmt_money(vu)}")

            linhas.append(f"P({int(round((prob or 0)*100))}%)")
            if prox:
                linhas.append(f"⏰ Próx.: {prox[:16]}")
                
            motivo = (motivo or "").strip()
            # Mostra motivo sempre que houver (você pode limitar só ao Perdido se preferir)
            if motivo:
                # Se quiser só no Perdido, use: if motivo and estagio == 'FECHADO_PERDIDO':
                linhas.append(f"📝 Nota: {self._short(motivo, 90)}")


            txt = "\n".join(linhas)

            it = QListWidgetItem(txt)
            it.setData(Qt.UserRole, int(opp_id))
            it.setSizeHint(QSize(240, max(68, 44 + 16*len(linhas))))
            # cor/estilo leve
            it.setForeground(Qt.black)

            lw = self.kanban_cols.get(estagio, self.kanban_cols["LEAD"])
            lw.addItem(it)

        # contadores no cabeçalho (se você tiver self._pipe_headers)
        if hasattr(self, "_pipe_headers"):
            for st, lbl in self._pipe_headers.items():
                lbl.setText(f"{ESTAGIO_LABEL.get(st, st)} ({self.kanban_cols[st].count()})")



    def _mover_oportunidade(self, opp_id: int, novo_estagio: str):
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("UPDATE oportunidades SET estagio=?, dt_atualizado=? WHERE id=?", (novo_estagio, _now_str(), opp_id))
        if novo_estagio in ('FECHADO_GANHO','FECHADO_PERDIDO'):
            cur.execute("UPDATE oportunidades SET dt_fechado=? WHERE id=?", (_now_str(), opp_id))
        # cria próxima ação automática se não houver tarefa pendente
        if novo_estagio in ('LEAD','QUALIFICADO','PROPOSTA','NEGOCIACAO'):
            cur.execute("SELECT COUNT(*) FROM atividades WHERE oportunidade_id=? AND status_tarefa='PENDENTE'", (opp_id,))
            has = cur.fetchone()[0]
            if not has:
                desc = f"Definir próxima ação – {ESTAGIO_LABEL.get(novo_estagio, novo_estagio)}"
                when = (datetime.now() + timedelta(days=2)).strftime("%Y-%m-%d %H:%M:%S")
                cur.execute("INSERT INTO atividades(oportunidade_id, tipo, descricao, status_tarefa, dt_prevista, dt_criado) VALUES (?,?,?,?,?,?)",
                            (opp_id, 'TAREFA', desc, 'PENDENTE', when, _now_str()))
        con.commit(); con.close()
        self._load_pipeline()


    def _nova_oportunidade(self):
        dlg = QDialog(self); dlg.setWindowTitle("Nova Oportunidade")
        v = QVBoxLayout(dlg)

        # Linha superior: Lead, Título, Probabilidade (sem Valor estimado)
        top = QHBoxLayout()
        lead_cb = QComboBox()
        for lid, nome in self._listar_leads():
            lead_cb.addItem(nome, lid)
        ed_tit = QLineEdit(); ed_tit.setPlaceholderText("Título da oportunidade")
        prob_cb = QComboBox(); prob_cb.addItems(["0.2","0.3","0.5","0.7","0.9"]); prob_cb.setCurrentText("0.5")
        for w in (QLabel("Lead:"), lead_cb, QLabel("Título:"), ed_tit, QLabel("Prob:"), prob_cb):
            top.addWidget(w)
        top.addStretch(1)
        v.addLayout(top)

        # Tabela simples de serviços: Serviço | Valor (R$)
        tbl = QTableWidget(); tbl.setColumnCount(2)
        tbl.setHorizontalHeaderLabels(["Serviço","Valor (R$)"])
        tbl.verticalHeader().setVisible(False)
        tbl.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        v.addWidget(tbl)

        hb = QHBoxLayout()
        btn_add = QPushButton("Adicionar serviço"); btn_del = QPushButton("Excluir serviço")
        hb.addWidget(btn_add); hb.addWidget(btn_del); hb.addStretch(1)
        v.addLayout(hb)

        def add_row():
            r = tbl.rowCount(); tbl.insertRow(r)
            tbl.setItem(r, 0, QTableWidgetItem("ECG"))
            spv = QDoubleSpinBox(); spv.setRange(0, 1_000_000); spv.setDecimals(2); spv.setPrefix("R$ "); spv.setValue(0.00)
            tbl.setCellWidget(r, 1, spv)
        def del_row():
            r = tbl.currentRow()
            if r >= 0: tbl.removeRow(r)
        btn_add.clicked.connect(add_row); btn_del.clicked.connect(del_row)
        add_row()

        bb = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        v.addWidget(bb)

        def on_accept():
            lid = lead_cb.currentData()
            titulo = ed_tit.text().strip() or "Oportunidade"
            prob = float(prob_cb.currentText())

            con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
            cur.execute(
                "INSERT INTO oportunidades(lead_id, titulo, valor_estimado, prob, estagio, dt_criado, dt_atualizado) VALUES (?,?,?,?,?,?,?)",
                (lid, titulo, 0.0, prob, 'LEAD', _now_str(), _now_str())
            )
            opp_id = cur.lastrowid

            # grava serviços (quantidade=1 sempre)
            for r in range(tbl.rowCount()):
                srv = (tbl.item(r, 0).text() if tbl.item(r, 0) else "").strip()
                vu  = tbl.cellWidget(r, 1).value() if tbl.cellWidget(r, 1) else 0.0
                if srv or vu > 0:
                    cur.execute(
                        "INSERT INTO itens_oportunidade(oportunidade_id, servico, quantidade, valor_unit) VALUES (?,?,?,?)",
                        (opp_id, srv or "Serviço", 1, float(vu))
                    )
            con.commit(); con.close()
            self._load_pipeline()
            dlg.accept()

        bb.accepted.connect(on_accept)
        bb.rejected.connect(dlg.reject)
        dlg.exec_()




    def _listar_leads(self) -> List[Tuple[int, str]]:
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("SELECT id, COALESCE(empresa_raw, empresa_norm, 'Lead') FROM leads ORDER BY dt_atualizado DESC LIMIT 500")
        rows = [(int(i), str(n)) for (i, n) in cur.fetchall()]
        con.close(); return rows

    # ----------------- LEADS -----------------
    def _montar_leads(self):
        root = QVBoxLayout(self.tab_leads); root.setContentsMargins(12,12,12,12); root.setSpacing(8)

        form = QHBoxLayout()
        self.ed_leads_busca = QLineEdit(); self.ed_leads_busca.setPlaceholderText("Buscar… (empresa, contato, cidade, tags)")
        self.ed_leads_busca.textChanged.connect(lambda: self._load_leads(self.ed_leads_busca.text().strip()))
        form.addWidget(self.ed_leads_busca)

        self.ed_empresa = QLineEdit(); self.ed_empresa.setPlaceholderText("Empresa")
        self.ed_contato = QLineEdit(); self.ed_contato.setPlaceholderText("Contato")
        self.ed_email   = QLineEdit(); self.ed_email.setPlaceholderText("Email")
        self.ed_tel     = QLineEdit(); self.ed_tel.setPlaceholderText("Telefone")
        self.ed_cidade  = QLineEdit(); self.ed_cidade.setPlaceholderText("Cidade")
        self.cb_uf      = QComboBox(); self.cb_uf.addItems(['','AC','AL','AP','AM','BA','CE','DF','ES','GO','MA','MT','MS','MG','PA','PB','PR','PE','PI','RJ','RN','RS','RO','RR','SC','SP','SE','TO'])
        self.cb_lista   = QComboBox(); self.cb_lista.addItems(['FRIA','MORNA','QUENTE'])
        self.ed_tags    = QLineEdit(); self.ed_tags.setPlaceholderText("tags, separadas por vírgula")
        btn_add = QPushButton("Adicionar Lead"); btn_add.clicked.connect(self._add_lead)
        btn_import = QPushButton("Importar CSV…"); btn_import.clicked.connect(self._importar_csv)

        for w in (self.ed_empresa, self.ed_contato, self.ed_email, self.ed_tel, self.ed_cidade, self.cb_uf, self.cb_lista, self.ed_tags, btn_add, btn_import):
            form.addWidget(w)
        form.addStretch(1)
        root.addLayout(form)

        # Tabela
        self.tbl_leads = QTableWidget(); self.tbl_leads.setColumnCount(10)
        self.tbl_leads.setHorizontalHeaderLabels(["ID","Empresa","Contato","Email","Telefone","Cidade","UF","Lista","Score","Tags"])
        self.tbl_leads.verticalHeader().setVisible(False)
        self.tbl_leads.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.tbl_leads.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tbl_leads.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tbl_leads.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.tbl_leads.setSortingEnabled(True)
        self.tbl_leads.setContextMenuPolicy(Qt.CustomContextMenu)
        self.tbl_leads.customContextMenuRequested.connect(self._menu_leads)
        self.tbl_leads.itemDoubleClicked.connect(lambda _i: self._editar_lead())

        root.addWidget(self.tbl_leads)

        btnrow = QHBoxLayout()
        btn_new_opp = QPushButton("Criar Oportunidade p/ Lead Selecionado"); btn_new_opp.clicked.connect(self._quick_create_opp_from_selected)
        btn_del = QPushButton("Excluir Lead"); btn_del.clicked.connect(self._del_lead)
        btnrow.addWidget(btn_new_opp); btnrow.addSpacing(12); btnrow.addWidget(btn_del); btnrow.addStretch(1)
        root.addLayout(btnrow)

        self._load_leads()

    def _add_lead(self):
        empresa = self.ed_empresa.text().strip()
        if not empresa:
            QMessageBox.information(self, "Leads", "Informe ao menos o nome da empresa."); return
        contato = self.ed_contato.text().strip(); email = self.ed_email.text().strip(); tel = self.ed_tel.text().strip()
        cidade  = self.ed_cidade.text().strip(); uf = self.cb_uf.currentText().strip() or None
        lista   = self.cb_lista.currentText(); tags = self.ed_tags.text().strip()
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute(
            """
            INSERT INTO leads(empresa_raw, empresa_norm, contato_nome, contato_email, contato_tel, cidade, uf, origem, lista, score, tags, dt_criado, dt_atualizado)
            VALUES (?,?,?,?,?,?,?,?,?,?,?, ?, ?)
            """,
            (empresa, _normalize_empresa(empresa), contato, email, tel, cidade, uf, 'Manual', lista, 0, tags, _now_str(), _now_str())
        )
        con.commit(); con.close()
        self._last_new_lead_id = cur.lastrowid
        self._clear_lead_form()
        self._load_leads()
        self._refresh_leads_combos()
        self._load_tasks()
        self._load_pipeline()
        self._calc_kpis()


    def _clear_lead_form(self):
        for w in (self.ed_empresa, self.ed_contato, self.ed_email, self.ed_tel, self.ed_cidade, self.ed_tags): w.clear()
        self.cb_uf.setCurrentIndex(0); self.cb_lista.setCurrentIndex(0)

    def _load_leads(self, search: str = ""):
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        if search:
            like = f"%{search}%"
            cur.execute("""
                SELECT id, empresa_raw, contato_nome, contato_email, contato_tel, cidade, uf, lista, score, COALESCE(tags,'')
                FROM leads
                WHERE COALESCE(empresa_raw,'') LIKE ? OR COALESCE(contato_nome,'') LIKE ?
                OR COALESCE(cidade,'') LIKE ? OR COALESCE(uf,'') LIKE ? OR COALESCE(tags,'') LIKE ?
                ORDER BY dt_atualizado DESC
            """, (like, like, like, like, like))
        else:
            cur.execute("SELECT id, empresa_raw, contato_nome, contato_email, contato_tel, cidade, uf, lista, score, COALESCE(tags,'') FROM leads ORDER BY dt_atualizado DESC")
        rows = cur.fetchall(); con.close()
        self.tbl_leads.setRowCount(0)
        for r in rows:
            row = self.tbl_leads.rowCount(); self.tbl_leads.insertRow(row)
            for c, val in enumerate(r):
                it = QTableWidgetItem(str(val if val is not None else ''))
                # pinta a “Lista” (coluna 7) para ficar mais intuitivo
                if c == 7:
                    cor = {"FRIA":"#e2e8f0","MORNA":"#fde68a","QUENTE":"#fecaca"}.get(str(val).upper(), "#e2e8f0")
                    it.setBackground(Qt.yellow if str(val).upper()=="QUENTE" else Qt.white)
                    it.setBackground(Qt.white)  # evita amarelo berrante
                    it.setData(Qt.BackgroundRole, None)
                    it.setForeground(Qt.black)
                    it.setToolTip("Temperatura do lead")
                self.tbl_leads.setItem(row, c, it)
                
    def _menu_leads(self, pos):
        row = self.tbl_leads.currentRow()
        if row < 0: return
        menu = QMenu(self)
        act_edit = QAction("Editar lead", self); act_del = QAction("Excluir lead", self)
        act_edit.triggered.connect(self._editar_lead)
        act_del.triggered.connect(self._del_lead)
        menu.addAction(act_edit); menu.addAction(act_del)
        menu.exec_(self.tbl_leads.viewport().mapToGlobal(pos))

    def _editar_lead(self):
        row = self.tbl_leads.currentRow()
        if row < 0: return
        lid = int(self.tbl_leads.item(row, 0).text())
        # pega dados atuais
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("SELECT empresa_raw, contato_nome, contato_email, contato_tel, cidade, uf, lista, tags FROM leads WHERE id=?", (lid,))
        r = cur.fetchone(); con.close()
        if not r: return
        emp, cont, email, tel, cid, uf, lista, tags = [str(x or "") for x in r]

        # mini “form” no topo (como nos outros diálogos do módulo)
        dlg = QWidget(self); lay = QHBoxLayout(dlg)
        ed_emp = QLineEdit(emp); ed_cont = QLineEdit(cont); ed_email = QLineEdit(email); ed_tel = QLineEdit(tel)
        ed_cid = QLineEdit(cid); cb_uf = QComboBox(); cb_uf.addItems(['','AC','AL','AP','AM','BA','CE','DF','ES','GO','MA','MT','MS','MG','PA','PB','PR','PE','PI','RJ','RN','RS','RO','RR','SC','SP','SE','TO']); cb_uf.setCurrentText(uf)
        cb_lista = QComboBox(); cb_lista.addItems(['FRIA','MORNA','QUENTE']); cb_lista.setCurrentText(lista or 'FRIA')
        ed_tags = QLineEdit(tags)
        btn_ok = QPushButton("Salvar"); btn_cancel = QPushButton("Cancelar")
        for w in (QLabel("Empresa:"), ed_emp, QLabel("Contato:"), ed_cont, QLabel("Email:"), ed_email, QLabel("Tel:"), ed_tel,
                QLabel("Cidade:"), ed_cid, QLabel("UF:"), cb_uf, QLabel("Lista:"), cb_lista, QLabel("Tags:"), ed_tags, btn_ok, btn_cancel):
            lay.addWidget(w)
        lay.addStretch(1)
        def do_save():
            con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
            cur.execute("""
                UPDATE leads SET empresa_raw=?, empresa_norm=?, contato_nome=?, contato_email=?, contato_tel=?, cidade=?, uf=?, lista=?, tags=?, dt_atualizado=?
                WHERE id=?
            """, (ed_emp.text().strip(), _normalize_empresa(ed_emp.text()), ed_cont.text().strip(), ed_email.text().strip(), ed_tel.text().strip(),
                ed_cid.text().strip(), cb_uf.currentText() or None, cb_lista.currentText(), ed_tags.text().strip(), _now_str(), lid))
            con.commit(); con.close()

            # >>> ATUALIZA TUDO <<<
            self._last_new_lead_id = lid                 # faz os combos selecionarem este lead
            self._load_leads(self.ed_leads_busca.text().strip())
            self._refresh_leads_combos()                 # repopula combos nas abas Tarefas/Cadências/Mensagens
            self._load_pipeline()                        # recarrega cards com o novo nome do lead
            self._load_tasks()                           # tabela de tarefas mostra o novo nome
            self._calc_kpis()                            # KPIs/Relatórios (se contar por lead) ficam coerentes

            dlg.setParent(None); dlg.deleteLater()

        btn_ok.clicked.connect(do_save); btn_cancel.clicked.connect(lambda: (dlg.setParent(None), dlg.deleteLater()))
        self.tab_leads.layout().insertWidget(1, dlg)



    def _quick_create_opp_from_selected(self):
        row = self.tbl_leads.currentRow()
        if row < 0:
            QMessageBox.information(self, "Leads", "Selecione um lead na tabela."); return
        lid = int(self.tbl_leads.item(row, 0).text())
        empresa = self.tbl_leads.item(row, 1).text()
        titulo = f"Proposta para {empresa}"
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute(
            "INSERT INTO oportunidades(lead_id, titulo, valor_estimado, prob, estagio, dt_criado, dt_atualizado) VALUES (?,?,?,?,?,?,?)",
            (lid, titulo, 0.0, 0.5, 'LEAD', _now_str(), _now_str())
        )
        con.commit(); con.close()
        QMessageBox.information(self, "Oportunidades", "Oportunidade criada no estágio LEAD.")
        self._load_pipeline()

    def _del_lead(self):
        row = self.tbl_leads.currentRow()
        if row < 0: return
        lid = int(self.tbl_leads.item(row, 0).text())
        if QMessageBox.question(self, "Excluir Lead", "Excluir o lead selecionado? Isto removerá oportunidades e atividades vinculadas.") != QMessageBox.Yes:
            return
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("DELETE FROM leads WHERE id=?", (lid,))
        con.commit(); con.close()
        self._load_leads(); self._load_pipeline(); self._load_tasks()

    # ----------------- TAREFAS -----------------
    def _montar_tarefas(self):
        root = QVBoxLayout(self.tab_tasks); root.setContentsMargins(12,12,12,12); root.setSpacing(8)
        # Criar tarefa
        frm = QHBoxLayout()
        self.cb_task_lead = QComboBox()
        for lid, nome in self._listar_leads(): self.cb_task_lead.addItem(nome, lid)
        self.cb_task_tipo = QComboBox(); self.cb_task_tipo.addItems(['TAREFA','EMAIL','WHATSAPP','LIGACAO','REUNIAO','ANOTACAO'])
        self.ed_task_desc = QLineEdit(); self.ed_task_desc.setPlaceholderText("Descrição da tarefa")
        self.dt_task_prev = QDateTimeEdit(QDateTime.currentDateTime().addDays(2)); self.dt_task_prev.setCalendarPopup(True)
        self.cb_task_filter = QComboBox(); self.cb_task_filter.addItems(["Pendentes","Feitas","Todas"])
        self.cb_task_filter.currentIndexChanged.connect(self._load_tasks)
        frm.addWidget(QLabel("Mostrar:")); frm.addWidget(self.cb_task_filter)

        btn_add = QPushButton("Adicionar"); btn_add.clicked.connect(self._add_task)
        for w in (QLabel("Lead:"), self.cb_task_lead, QLabel("Tipo:"), self.cb_task_tipo, self.ed_task_desc, QLabel("Quando:"), self.dt_task_prev, btn_add):
            frm.addWidget(w)
        frm.addStretch(1)
        root.addLayout(frm)

        # Tabela de tarefas
        self.tbl_tasks = QTableWidget(); self.tbl_tasks.setColumnCount(8)
        self.tbl_tasks.setHorizontalHeaderLabels(["ID","Lead","Tipo","Descrição","Status","Prevista","Realizada","Resp."])
        self.tbl_tasks.verticalHeader().setVisible(False)
        self.tbl_tasks.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.tbl_tasks.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tbl_tasks.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tbl_tasks.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.tbl_tasks.setContextMenuPolicy(Qt.CustomContextMenu)
        self.tbl_tasks.customContextMenuRequested.connect(self._menu_tasks)

        root.addWidget(self.tbl_tasks)

        actions = QHBoxLayout()
        btn_done = QPushButton("Concluir"); btn_done.clicked.connect(self._mark_task_done)
        btn_defer = QPushButton("Adiantar +2 dias"); btn_defer.clicked.connect(self._defer_task)
        actions.addWidget(btn_done); actions.addWidget(btn_defer); actions.addStretch(1)
        root.addLayout(actions)

        self._load_tasks()
        
    def _menu_tasks(self, pos):
        row = self.tbl_tasks.currentRow()
        if row < 0: return
        menu = QMenu(self)
        act_done = QAction("Concluir", self); act_del = QAction("Excluir tarefa", self)
        act_done.triggered.connect(self._mark_task_done)
        act_del.triggered.connect(self._del_task)
        menu.addAction(act_done); menu.addAction(act_del)
        menu.exec_(self.tbl_tasks.viewport().mapToGlobal(pos))

    def _del_task(self):
        row = self.tbl_tasks.currentRow()
        if row < 0: return
        tid = int(self.tbl_tasks.item(row, 0).text())
        if QMessageBox.question(self, "Excluir Tarefa", "Confirmar exclusão desta tarefa?") != QMessageBox.Yes:
            return
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("DELETE FROM atividades WHERE id=?", (tid,))
        con.commit(); con.close()
        self._load_tasks()


    def _add_task(self):
        lid = self.cb_task_lead.currentData()
        tipo = self.cb_task_tipo.currentText()
        desc = self.ed_task_desc.text().strip() or tipo
        dtp  = self.dt_task_prev.dateTime().toString("yyyy-MM-dd HH:mm:ss")
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute(
            "INSERT INTO atividades(lead_id, tipo, descricao, status_tarefa, dt_prevista, dt_criado) VALUES (?,?,?,?,?,?)",
            (lid, tipo, desc, 'PENDENTE', dtp, _now_str())
        )
        con.commit(); con.close()
        self.ed_task_desc.clear(); self._load_tasks()

    def _load_tasks(self):
        status = self.cb_task_filter.currentText() if hasattr(self, "cb_task_filter") else "Pendentes"
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        base = """
            SELECT a.id, COALESCE(l.empresa_raw,l.empresa_norm,''), a.tipo, a.descricao, a.status_tarefa,
                COALESCE(a.dt_prevista,''), COALESCE(a.dt_realizada,''), COALESCE(a.responsavel,'')
            FROM atividades a LEFT JOIN leads l ON l.id=a.lead_id
        """
        if status == "Pendentes":
            base += " WHERE a.status_tarefa='PENDENTE'"
        elif status == "Feitas":
            base += " WHERE a.status_tarefa='FEITA'"
        base += " ORDER BY COALESCE(a.dt_prevista, a.dt_criado) ASC"
        cur.execute(base); rows = cur.fetchall(); con.close()
        self.tbl_tasks.setRowCount(0)
        for r in rows:
            row = self.tbl_tasks.rowCount(); self.tbl_tasks.insertRow(row)
            for c, val in enumerate(r): self.tbl_tasks.setItem(row, c, QTableWidgetItem(str(val or '')))


    def _mark_task_done(self):
        row = self.tbl_tasks.currentRow()
        if row < 0: return
        tid = int(self.tbl_tasks.item(row, 0).text())
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("UPDATE atividades SET status_tarefa='FEITA', dt_realizada=? WHERE id=?", (_now_str(), tid))
        con.commit(); con.close(); self._load_tasks()

    def _defer_task(self):
        row = self.tbl_tasks.currentRow()
        if row < 0: return
        tid = int(self.tbl_tasks.item(row, 0).text())
        prev = self.tbl_tasks.item(row, 5).text() or _now_str()
        try:
            base_dt = datetime.strptime(prev, "%Y-%m-%d %H:%M:%S")
        except Exception:
            base_dt = datetime.now()
        new_dt = (base_dt + timedelta(days=2)).strftime("%Y-%m-%d %H:%M:%S")
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("UPDATE atividades SET dt_prevista=? WHERE id=?", (new_dt, tid))
        con.commit(); con.close(); self._load_tasks()

    # ----------------- CADÊNCIAS -----------------
    def _montar_cadencias(self):
        root = QVBoxLayout(self.tab_cad); root.setContentsMargins(12,12,12,12); root.setSpacing(8)
        top = QHBoxLayout()
        self.cb_cad_lead = QComboBox();
        for lid, nome in self._listar_leads(): self.cb_cad_lead.addItem(nome, lid)
        self.cb_cadencias = QComboBox(); self._refresh_cadencias_combo()
        btn_apply = QPushButton("Aplicar ao Lead"); btn_apply.clicked.connect(self._apply_cadence)
        btn_new   = QPushButton("Nova Cadência…"); btn_new.clicked.connect(self._nova_cadencia)
        for w in (QLabel("Lead:"), self.cb_cad_lead, QLabel("Cadência:"), self.cb_cadencias, btn_apply, btn_new): top.addWidget(w)
        top.addStretch(1); root.addLayout(top)

        self.txt_cad = QTextEdit(); self.txt_cad.setReadOnly(True)
        root.addWidget(self.txt_cad)
        self.cb_cadencias.currentIndexChanged.connect(self._show_cadencia_text)
        self._show_cadencia_text()

    def _refresh_cadencias_combo(self):
        self.cb_cadencias.clear()
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("SELECT id, nome, passos_json FROM cadencias WHERE ativo=1 ORDER BY id DESC")
        self._cad_cache = cur.fetchall(); con.close()
        for cid, nome, _ in self._cad_cache:
            self.cb_cadencias.addItem(nome, cid)

    def _show_cadencia_text(self):
        idx = self.cb_cadencias.currentIndex()
        if idx < 0: self.txt_cad.setPlainText(""); return
        _, _, passos_json = self._cad_cache[idx]
        try:
            passos = json.loads(passos_json)
        except Exception:
            passos = []
        txt = "\n".join([f"D+{p.get('dias',0)}  {p.get('tipo','TAREFA')}  –  {p.get('descricao','')}" for p in passos])
        self.txt_cad.setPlainText(txt)

    def _apply_cadence(self):
        idx = self.cb_cadencias.currentIndex()
        if idx < 0: return
        lid = self.cb_cad_lead.currentData()
        passos = json.loads(self._cad_cache[idx][2]) if self._cad_cache[idx][2] else []
        base_dt = datetime.now()
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        for p in passos:
            dias = int(p.get('dias',0)); tipo = p.get('tipo','TAREFA'); desc = p.get('descricao','Tarefa')
            when = (base_dt + timedelta(days=dias)).strftime("%Y-%m-%d %H:%M:%S")
            cur.execute(
                "INSERT INTO atividades(lead_id, tipo, descricao, status_tarefa, dt_prevista, dt_criado) VALUES (?,?,?,?,?,?)",
                (lid, tipo, desc, 'PENDENTE', when, _now_str())
            )
        con.commit(); con.close()
        QMessageBox.information(self, "Cadências", "Cadência aplicada e tarefas criadas.")
        self._load_tasks()

    def _nova_cadencia(self):
        # mini-dialog embutido
        dlg = QWidget(self)
        lay = QHBoxLayout(dlg)
        ed_nome = QLineEdit(); ed_nome.setPlaceholderText("Nome da cadência")
        ed_json = QLineEdit(); ed_json.setPlaceholderText("Passos em JSON (ex.: [{\"dias\":0,\"tipo\":\"EMAIL\",\"descricao\":\"Apresentação\"}] )")
        btn_ok  = QPushButton("Salvar")
        for w in (ed_nome, ed_json, btn_ok): lay.addWidget(w)
        lay.addStretch(1)
        def do_save():
            nome = ed_nome.text().strip() or "Cadência"
            try:
                passos = json.loads(ed_json.text()) if ed_json.text().strip() else []
            except Exception:
                QMessageBox.warning(self, "Cadências", "JSON inválido."); return
            con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
            cur.execute("INSERT INTO cadencias(nome, passos_json, ativo) VALUES (?,?,1)", (nome, json.dumps(passos, ensure_ascii=False)))
            con.commit(); con.close()
            self._refresh_cadencias_combo(); dlg.setParent(None); dlg.deleteLater()
        btn_ok.clicked.connect(do_save)
        self.tab_cad.layout().insertWidget(1, dlg)

    # ----------------- MENSAGENS -----------------
    def _montar_mensagens(self):
        root = QVBoxLayout(self.tab_msg); root.setContentsMargins(12,12,12,12); root.setSpacing(8)
        top = QHBoxLayout()
        self.cb_msg_lead = QComboBox();
        for lid, nome in self._listar_leads(): self.cb_msg_lead.addItem(nome, lid)
        self.cb_msg_tpl  = QComboBox();
        self.cb_msg_tpl.addItems([
            "Email – Apresentação",
            "WhatsApp – Follow-up",
            "Resumo – Proposta enviada"
        ])
        btn_gen = QPushButton("Gerar"); btn_gen.clicked.connect(self._gerar_msg)
        btn_copy = QPushButton("Copiar texto"); btn_copy.clicked.connect(self._copiar_msg)
        for w in (QLabel("Lead:"), self.cb_msg_lead, QLabel("Modelo:"), self.cb_msg_tpl, btn_gen, btn_copy): top.addWidget(w)
        top.addStretch(1); root.addLayout(top)

        self.txt_msg = QTextEdit(); root.addWidget(self.txt_msg, 1)

    def _gerar_msg(self):
        lid = self.cb_msg_lead.currentData()
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("SELECT empresa_raw, cidade, uf, contato_nome FROM leads WHERE id=?", (lid,))
        row = cur.fetchone(); con.close()
        empresa, cidade, uf, contato = row if row else ("Empresa","","","")
        modelo = self.cb_msg_tpl.currentText()
        if "Apresentação" in modelo:
            txt = (f"Olá {contato or ''}, tudo bem?\n\n"
                   f"Somos a Medical Laudos. Em {cidade or ''}/{uf or ''}, temos SLA de laudos ágil e painel online.\n"
                   f"Poderia te mostrar, em 10 min, como reduzimos o tempo e custo dos seus exames?\n\n"
                   f"Posso te ligar ainda essa semana?")
        elif "Follow-up" in modelo:
            txt = (f"Oi {contato or ''}! Conseguiu ver minha mensagem anterior sobre a {empresa}?\n"
                   f"Tenho agenda flexível e um comparativo pronto do cenário atual. Topa marcarmos?")
        else:
            txt = (f"{contato or 'Olá'}, conforme combinado, enviei a proposta para {empresa}.\n"
                   f"Fico à disposição para ajustes de SLA e valores. Quando posso te ligar?")
        self.txt_msg.setPlainText(txt)

    def _copiar_msg(self):
        from PyQt5.QtWidgets import QApplication
        QApplication.clipboard().setText(self.txt_msg.toPlainText())
        QMessageBox.information(self, "Mensagens", "Texto copiado para a área de transferência.")

    # ----------------- RELATÓRIOS -----------------
    def _montar_relatorios(self):
        root = QVBoxLayout(self.tab_rep); root.setContentsMargins(12,12,12,12); root.setSpacing(8)
        self.lbl_kpis = QLabel("–"); self.lbl_kpis.setStyleSheet("font-weight:600;color:#334155")
        btn = QPushButton("Recalcular"); btn.clicked.connect(self._calc_kpis)
        top = QHBoxLayout(); top.addWidget(self.lbl_kpis); top.addStretch(1); top.addWidget(btn)
        root.addLayout(top)
        self._calc_kpis()

    def _calc_kpis(self):
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute("SELECT COUNT(*) FROM leads"); leads = cur.fetchone()[0]
        cur.execute("SELECT COUNT(*) FROM oportunidades WHERE estagio NOT IN ('FECHADO_GANHO','FECHADO_PERDIDO')"); opps_abertas = cur.fetchone()[0]
        cur.execute("SELECT COALESCE(SUM(valor_estimado*prob),0) FROM oportunidades WHERE estagio NOT IN ('FECHADO_GANHO','FECHADO_PERDIDO')")
        forecast = float(cur.fetchone()[0] or 0)
        cur.execute("SELECT SUM(estagio='FECHADO_GANHO')*1.0, SUM(estagio IN ('FECHADO_GANHO','FECHADO_PERDIDO'))*1.0 FROM oportunidades")
        won, closed = cur.fetchone()
        winrate = (won/closed*100.0) if closed else 0.0
        cur.execute("SELECT COUNT(*) FROM atividades WHERE status_tarefa='PENDENTE'")
        tasks_pend = cur.fetchone()[0]
        con.close()
        txt = (f"Leads: {leads}  |  Oportunidades abertas: {opps_abertas}  |  Forecast: R$ {forecast:,.0f}  |  Win rate: {winrate:.1f}%  |  Tarefas pend.: {tasks_pend}")
        txt = txt.replace(",","X").replace(".",",").replace("X", ".")
        self.lbl_kpis.setText(txt)

    # ----------------- INTEGRAÇÃO EXTERNA -----------------
    def adicionar_lead_prepreenchido(self, dados: Dict[str, str]) -> int:
        """Cria um lead a partir de dados vindos de outros módulos. Retorna o ID."""
        empresa = dados.get('empresa') or dados.get('Empresa') or ''
        empresa_norm = _normalize_empresa(empresa)
        contato = dados.get('contato') or ''
        email   = dados.get('email') or ''
        tel     = dados.get('tel') or ''
        cidade  = dados.get('cidade') or ''
        uf      = dados.get('uf') or None
        origem  = dados.get('origem') or 'Analises'
        lista   = dados.get('lista') or 'FRIA'
        score   = float(dados.get('score') or 0)
        tags    = dados.get('tags') or ''
        con = sqlite3.connect(CAMINHO_BANCO); cur = con.cursor()
        cur.execute(
            """
            INSERT INTO leads(empresa_raw, empresa_norm, contato_nome, contato_email, contato_tel, cidade, uf, origem, lista, score, tags, dt_criado, dt_atualizado)
            VALUES (?,?,?,?,?,?,?,?,?,?,?, ?, ?)
            """,
            (empresa, empresa_norm, contato, email, tel, cidade, uf, origem, lista, score, tags, _now_str(), _now_str())
        )
        lead_id = cur.lastrowid
        con.commit(); con.close()
        self._last_new_lead_id = lead_id
        self._refresh_leads_combos()
        self._load_tasks()
        self._load_pipeline()
        self._calc_kpis()

        self._load_leads()
        return int(lead_id)

    # ----------------- ESTILO -----------------
    def _apply_style(self):
        self.setStyleSheet(
            """
            QTabWidget::pane { border: 1px solid #c7d7e5; border-radius: 8px; padding: 0; background: #f7f9fc; }
            QTabBar::tab { background: #e6edf5; color: #0f172a; border: 1px solid #d7e1ee; border-bottom: none; border-top-left-radius: 8px; border-top-right-radius: 8px; padding: 6px 12px; height: 30px; font-weight: 600; }
            QTabBar::tab:selected { background: #0ea5e9; color: #ffffff; border: 1px solid #0284c7; }
            QPushButton { padding:6px 10px; }
            QTableWidget { background:#ffffff; border:1px solid #e6eef5; border-radius:6px; }
            QTextEdit, QLineEdit, QComboBox { background:#ffffff; }
            """
        )
