from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QPushButton, QFileDialog,
    QTableWidget, QTableWidgetItem, QLabel, QHBoxLayout, QComboBox, QDialog, QDialogButtonBox, QFormLayout
)
import pandas as pd
import os
import sqlite3
from PyQt5.QtWidgets import QMessageBox
from PyQt5.QtWidgets import QLineEdit
from PyQt5.QtCore import QTimer
from PyQt5 import QtGui
from PyQt5.QtCore import Qt
import os
from PIL import Image

class CompetenciaDialog(QDialog):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Selecionar competência")
        self.setFixedSize(300, 150)

        layout = QFormLayout()
        self.combo_mes = QComboBox()
        self.combo_ano = QComboBox()

        self.combo_mes.addItems(["01", "02", "03", "04", "05", "06",
                                 "07", "08", "09", "10", "11", "12"])
        self.combo_ano.addItems(["2025", "2026", "2027", 
                                 "2028", "2029", "2030", "2031", "2032", "2033"])

        layout.addRow("Mês:", self.combo_mes)
        layout.addRow("Ano:", self.combo_ano)

        buttons = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        buttons.accepted.connect(self.accept)
        buttons.rejected.connect(self.reject)
        layout.addWidget(buttons)

        self.setLayout(layout)

    def get_competencia(self):
        return self.combo_mes.currentText(), self.combo_ano.currentText()

class ModuloTabelaRegistros(QWidget):
    def __init__(self):
        super().__init__()
        self.layout = QVBoxLayout(self)
        self.setLayout(self.layout)

        # ➕ Estilo uniforme para botões
        estilo_botao = """
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #2980b9;
            }
        """

        # Inicialização do filtro temporizado
        self.timer_filtro = QTimer()
        self.timer_filtro.setSingleShot(True)
        self.timer_filtro.timeout.connect(self._executar_filtro)

        # Status
        self.label_status = QLabel("Selecione uma competência para visualizar os registros.")
        self.botao_importar = QPushButton("Importar Planilha Excel")
        self.botao_importar.clicked.connect(self.importar_planilha)

        # Filtros por competência
        self.filtros = QHBoxLayout()
        self.combo_mes = QComboBox()
        self.combo_ano = QComboBox()
        self.combo_mes.addItems(["01", "02", "03", "04", "05", "06",
                                "07", "08", "09", "10", "11", "12"])
        self.combo_ano.addItems(["2025", "2026", "2027", 
                                 "2028", "2029", "2030", "2031", "2032", "2033"])

        self.botao_carregar = QPushButton("Carregar Competência")
        self.botao_carregar.clicked.connect(self.aplicar_filtro_competencia)

        self.botao_limpar_filtros = QPushButton("Limpar Filtros")
        self.botao_limpar_filtros.clicked.connect(self.limpar_filtros_colunas)

        self.botao_excluir = QPushButton("Excluir Competência")
        self.botao_excluir.clicked.connect(self.excluir_competencia)

        self.botao_exportar_excel = QPushButton("Exportar Excel")
        self.botao_exportar_excel.clicked.connect(self.exportar_para_excel)

        self.botao_excluir_selecionados = QPushButton("Excluir Selecionados")
        self.botao_excluir_selecionados.clicked.connect(self.excluir_selecionados)

        # Adiciona botões e campos ao layout de filtros
        self.filtros.addWidget(self.botao_limpar_filtros)
        self.filtros.addWidget(self.botao_excluir)
        self.filtros.addWidget(self.botao_exportar_excel)
        self.filtros.addWidget(self.botao_excluir_selecionados)
        self.filtros.addWidget(QLabel("Visualizar competência:"))
        self.filtros.addWidget(self.combo_mes)
        self.filtros.addWidget(self.combo_ano)
        self.filtros.addWidget(self.botao_carregar)

        # Tabela e filtros por coluna
        self.tabela = QTableWidget()
        self.tabela.verticalHeader().setVisible(False)
        self.tabela.setSortingEnabled(True)
        self.filtros_colunas_layout = QHBoxLayout()
        self.filtros_colunas_widgets = []
        self.layout.insertLayout(3, self.filtros_colunas_layout)
        self.dados = pd.DataFrame()

        # Adiciona widgets principais ao layout
        self.layout.addWidget(self.label_status)
        self.layout.addWidget(self.botao_importar)
        self.layout.addLayout(self.filtros)
        self.layout.addWidget(self.tabela)

        # Rodapé
        self.label_resumo = QLabel("<b>Registros: 0 | Convênios: 0 | Valor Convênio: R$ 0,00 | Valor Médico: R$ 0,00 | Registros Duplicados: 0</b>")
        self.layout.addWidget(self.label_resumo)

        # Botão para atualizar valores médicos
        self.botao_atualizar_valores = QPushButton("🔄 Atualizar Valores Médicos")
        self.botao_atualizar_valores.clicked.connect(self.atualizar_valores_medicos_salvos)
        layout_botao = QHBoxLayout()
        layout_botao.addWidget(self.botao_atualizar_valores)
        self.layout.insertLayout(0, layout_botao)

        # ✅ Aplicar estilo aos botões DEPOIS de criá-los
        botoes = [
            self.botao_importar,
            self.botao_limpar_filtros,
            self.botao_excluir,
            self.botao_exportar_excel,
            self.botao_excluir_selecionados,
            self.botao_carregar,
            self.botao_atualizar_valores
        ]
        for botao in botoes:
            botao.setStyleSheet(estilo_botao)


    def atualizar_valores_medicos_salvos(self):
        try:
            conn = sqlite3.connect('db/sistema_financeiro.db')
            cursor = conn.cursor()

            for i in range(len(self.dados)):
                medico = self.dados.iloc[i]["Médico"]
                convenio = self.dados.iloc[i]["Convenio"]
                procedimento = self.dados.iloc[i]["Procedimento"]
                cod_registro = self.dados.iloc[i]["Cod Registro"]

                cursor.execute("""
                    SELECT mp.valor_medico
                    FROM medico_procedimentos mp
                    JOIN medicos m ON mp.medico_id = m.id
                    WHERE m.nome = ? AND mp.convenio = ? AND mp.procedimento = ?
                """, (medico, convenio, procedimento))

                resultado = cursor.fetchone()
                if resultado:
                    novo_valor = float(resultado[0])
                    self.dados.at[i, "Valor Médico"] = f"R$ {novo_valor:,.2f}".replace(".", "#").replace(",", ".").replace("#", ",")

                    cursor.execute("""
                        UPDATE registros_financeiros
                        SET `Valor Médico` = ?
                        WHERE `Cod Registro` = ?
                    """, (novo_valor, cod_registro))

            conn.commit()
            conn.close()

            self.atualizar_tabela(self.dados)
            self.criar_filtros_colunas(self.dados)  # ✅ Adicione isso SE não estiver já dentro de atualizar_tabela
            self.label_status.setText("Valores médicos atualizados com sucesso.")

        except Exception as e:
            self.label_status.setText(f"Erro ao atualizar valores médicos: {e}")
            print(f"[ERRO] {e}")




    def recarregar_tabela(self):
        try:
            conn = sqlite3.connect('db/sistema_financeiro.db')
            df = pd.read_sql_query("SELECT * FROM registros_financeiros", conn)
            conn.close()

            # Conversão e formatação dos valores
            df = self.tratar_duplicados(df)
            df = self.formatar_valores_monetarios(df)

            self.dados = df
            self.atualizar_tabela(df)
            self.criar_filtros_colunas(df)

        except Exception as e:
            print(f"[ERRO ao recarregar tabela] {e}")



    def obter_dados_visiveis(self):
        linhas = self.tabela.rowCount()
        colunas = self.tabela.columnCount()
        nomes_colunas = [self.tabela.horizontalHeaderItem(j).text() for j in range(colunas)]
        dados = []

        for i in range(linhas):
            linha = []
            for j in range(colunas):
                item = self.tabela.item(i, j)
                linha.append(item.text() if item else "")
            dados.append(linha)

        return pd.DataFrame(dados, columns=nomes_colunas)


    def importar_planilha(self):
        caminho, _ = QFileDialog.getOpenFileName(self, "Selecionar planilha Excel", "", "Excel Files (*.xlsx *.xls)")
        if caminho:
            try:
                colunas_personalizadas = [
                    "Cod Registro", "Data Exame", "Nome", "Empresa", "Convenio",
                    "Tipo de Exame", "Procedimento", "Médico", "Data Cadastro", "Data Recebimento",
                    "Valor Médico", "Valor Convenio"
                ]
                df = pd.read_excel(
                    caminho,
                    header=None,
                    skiprows=2,
                    names=colunas_personalizadas,
                    dtype=str,
                    engine='openpyxl'
                )

                self.df_original = df.copy()
                self.dados = df.copy()

                # Padroniza datas com ou sem hora
                for col in ["Data Exame", "Data Cadastro", "Data Recebimento"]:
                    if col in df.columns:
                        df[col] = df[col].astype(str).str.strip()

                # Convertendo Valor Convenio (mantém como estava)
                df["Valor Convenio"] = pd.to_numeric(df["Valor Convenio"], errors='coerce').fillna(0)

                # Substitui Valor Médico pelo valor vindo do módulo 2 (tabela medico_procedimentos)
                conn = sqlite3.connect('db/sistema_financeiro.db')
                cursor = conn.cursor()
                valores_medicos = []

                for _, row in df.iterrows():
                    nome_medico = str(row.get("Médico", "")).strip()
                    procedimento = str(row.get("Procedimento", "")).strip()
                    convenio = str(row.get("Convenio", "")).strip()

                    cursor.execute("""
                        SELECT mp.valor_medico
                        FROM medico_procedimentos mp
                        JOIN medicos m ON mp.medico_id = m.id
                        WHERE m.nome = ? AND mp.procedimento = ? AND mp.convenio = ?
                    """, (nome_medico, procedimento, convenio))

                    resultado = cursor.fetchone()
                    valor = resultado[0] if resultado else 0.0
                    valores_medicos.append(valor)

                df["Valor Médico"] = valores_medicos

                df.rename(columns=lambda x: x.strip().replace("Û", "e").replace("ê", "e"), inplace=True)

                mes, ano = self.abrir_dialogo_competencia()
                if not mes or not ano:
                    self.label_status.setText("Importação cancelada pelo usuário.")
                    return

                competencia = f"{mes}/{ano}"
                df["Competência"] = competencia

                # Padronizar datas para o formato dd/mm/yyyy
                def padronizar_data(data):
                    if pd.isnull(data):
                        return ""
                    try:
                        return pd.to_datetime(str(data)).strftime("%d/%m/%Y")
                    except:
                        return str(data).strip()

                for coluna_data in ["Data Exame", "Data Cadastro", "Data Recebimento"]:
                    if coluna_data in df.columns:
                        df[coluna_data] = df[coluna_data].apply(padronizar_data)

                # Salva no banco
                df.to_sql("registros_financeiros", conn, if_exists='append', index=False)

                # Lê do banco apenas os dados da competência importada
                df_banco = pd.read_sql_query(
                    "SELECT * FROM registros_financeiros WHERE Competência = ?", conn, params=(competencia,)
                )

                df_banco = self.tratar_duplicados(df_banco)  # ✅ Aqui é onde as colunas são recriadas
                conn.close()

                # Formatação para exibição
                df_banco = self.formatar_valores_monetarios(df_banco)


                ordem_colunas = [
                    "Cod Registro", "Data Exame", "Nome", "Empresa", "Convenio",
                    "Tipo de Exame", "Procedimento", "Médico",
                    "Data Cadastro", "Data Recebimento", "Valor Médico", "Valor Convenio",
                    "Duplicado", "NaoDuplicadoOIT", "Competência"
                ]
                df_banco = df_banco[[col for col in ordem_colunas if col in df_banco.columns]]

                self.dados = df_banco
                print(df_banco.columns.tolist())
                self.atualizar_tabela(df_banco)
                self.label_status.setText(f"Planilha carregada: {os.path.basename(caminho)}")
                self.criar_filtros_colunas(df_banco)

            except Exception as e:
                self.label_status.setText(f"Erro ao carregar planilha: {e}")
                print(f"[ERRO] {e}")



    def aplicar_filtro_competencia(self):
        mes = self.combo_mes.currentText()
        ano = self.combo_ano.currentText()

        if mes == "Todos" or ano == "Todos":
            self.label_status.setText("Selecione mês e ano válidos para carregar a competência.")
            self.tabela.clear()
            self.tabela.setRowCount(0)
            self.tabela.setColumnCount(0)
            return

        try:
            # 🧹 Limpa os filtros das colunas (sem disparar sinais)
            if hasattr(self, 'filtros_colunas_widgets'):
                for filtro in self.filtros_colunas_widgets:
                    filtro.blockSignals(True)
                    filtro.setText("")
                    filtro.blockSignals(False)

            # ⚠️ Remove ordenação ativa do cabeçalho (visualmente e internamente)
            self.tabela.setSortingEnabled(False)
            self.tabela.horizontalHeader().setSortIndicator(-1, Qt.AscendingOrder)
            self.tabela.setSortingEnabled(True)

            # 🔍 Busca a competência no banco
            conn = sqlite3.connect("db/sistema_financeiro.db")
            competencia = f"{mes}/{ano}"
            query = "SELECT * FROM registros_financeiros WHERE Competência = ?"
            df = pd.read_sql_query(query, conn, params=(competencia,))
            conn.close()

            # ⚙️ Trata duplicados e valores monetários
            df = self.tratar_duplicados(df)
            df = self.formatar_valores_monetarios(df)

            # 🧠 Salva e atualiza na tela
            self.dados = df
            self.df_original = df.copy()
            self.atualizar_tabela(df)
            self.label_status.setText(f"Registros da competência {competencia} carregados.")

            self.criar_filtros_colunas(df)

        except Exception as e:
            self.label_status.setText(f"Erro ao aplicar filtro: {e}")



    def atualizar_tabela(self, df):
        self.tabela.clear()

        if df.empty:
            self.tabela.setRowCount(0)
            self.tabela.setColumnCount(0)
            return

        df = self.tratar_duplicados(df)
        # print("\n[DEBUG] Verificando NaoDuplicadoOIT:")
        # print(df[df["Nome"].str.contains("MARCIO JOSE CADA", case=False)][["Nome", "Procedimento", "Médico", "NaoDuplicadoOIT", "Duplicado"]])

        colunas = df.columns.tolist()
        self.tabela.setColumnCount(len(colunas))
        self.tabela.setRowCount(len(df))  # sem linha de filtros dentro da tabela
        self.tabela.setHorizontalHeaderLabels(colunas)

        self.filtros_colunas_widgets = []  # mantido limpo

        for i in range(len(df)):
            for j in range(len(colunas)):
                valor = df.iat[i, j]
                nome_coluna = colunas[j]

                if nome_coluna in ["Valor Médico", "Valor Convenio"]:
                    try:
                        if str(valor).strip().startswith("R$"):
                            # Já está formatado corretamente
                            texto = str(valor).strip()
                        else:
                            valor_float = float(str(valor).replace(",", ".").strip())
                            texto = f"R$ {valor_float:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
                    except:
                        texto = str(valor)

                elif nome_coluna == "Data Exame":
                    texto = str(valor).strip()  # Não converte mais!


                elif nome_coluna in ["Data Cadastro", "Data Recebimento"]:
                    texto = str(valor).strip()                        

                else:
                    texto = str(valor)

                item = QTableWidgetItem(texto)
                item.setTextAlignment(Qt.AlignCenter)

                # 🌿 Primeiro, aplicar verde se for exceção válida
                if "NaoDuplicadoOIT" in df.columns and str(df.iat[i, df.columns.get_loc("NaoDuplicadoOIT")]) == "True":
                    item.setBackground(QtGui.QColor("#C8E6C9"))  # Verde claro

                # ⚠️ Depois, aplicar amarelo se for duplicado (mas não exceção)
                elif "Duplicado" in df.columns and str(df.iat[i, df.columns.get_loc("Duplicado")]) == "True":
                    item.setBackground(QtGui.QColor("#FFF59D"))  # Amarelo claro

                self.tabela.setItem(i, j, item)


        self.tabela.resizeColumnsToContents()
        self.atualizar_resumo(df)

        if "Duplicado" in colunas:
            idx = colunas.index("Duplicado")
            #self.tabela.setColumnHidden(idx, True)

        colunas_ocultas = ["Competencia"]
        for nome_coluna in colunas_ocultas:
            if nome_coluna in colunas:
                idx = colunas.index(nome_coluna)
                self.tabela.setColumnHidden(idx, True)




    def abrir_dialogo_competencia(self):
        dialog = CompetenciaDialog()
        if dialog.exec_() == QDialog.Accepted:
            return dialog.get_competencia()
        return None, None
    

    # def carregar_dados_do_banco(self):
    #     try:
    #         conn = sqlite3.connect("db/sistema_financeiro.db")
    #         df = pd.read_sql_query("SELECT * FROM registros_financeiros", conn)
    #         conn.close()
    #         self.dados = df
    #         self.atualizar_tabela(df)
    #         self.label_status.setText("Dados carregados do banco de dados.")
    #     except Exception as e:
    #         self.label_status.setText(f"Erro ao carregar dados do banco: {e}")

    def excluir_competencia(self):
        mes = self.combo_mes.currentText()
        ano = self.combo_ano.currentText()

        if mes == "Todos" or ano == "Todos":
            QMessageBox.warning(self, "Atenção", "Selecione mês e ano válidos para excluir a competência.")
            return

        competencia = f"{mes}/{ano}"

        confirmacao = QMessageBox.question(
            self,
            "Confirmar Exclusão",
            f"Tem certeza que deseja excluir TODOS os registros da competência {competencia}?",
            QMessageBox.Yes | QMessageBox.No
        )

        if confirmacao == QMessageBox.Yes:
            try:
                conn = sqlite3.connect("db/sistema_financeiro.db")
                cursor = conn.cursor()
                cursor.execute("DELETE FROM registros_financeiros WHERE Competência = ?", (competencia,))
                conn.commit()
                conn.close()

                self.dados = pd.DataFrame()
                self.atualizar_tabela(self.dados)
                self.label_status.setText(f"Competência {competencia} excluída com sucesso.")
            except Exception as e:
                QMessageBox.critical(self, "Erro", f"Erro ao excluir dados: {e}")

    def aplicar_filtros_colunas(self):
        self.timer_filtro.start(500)  # Só inicia o timer


    def criar_filtros_colunas(self, df):
        # 🔴 Limpa os widgets antigos do layout
        while self.filtros_colunas_layout.count():
            item = self.filtros_colunas_layout.takeAt(0)
            widget = item.widget()
            if widget:
                widget.deleteLater()

        self.filtros_colunas_widgets = []

        # 🔵 Cria os novos filtros
        for j, col in enumerate(df.columns):
            filtro = QLineEdit()
            filtro.setPlaceholderText(f"Filtrar {col}")
            filtro.textChanged.connect(self.aplicar_filtros_colunas)
            self.filtros_colunas_layout.addWidget(filtro)
            self.filtros_colunas_widgets.append(filtro)

    def _executar_filtro(self):
        if self.dados.empty:
            return

        self.tabela.setUpdatesEnabled(False)

        # ⛔ Desativa ordenação visual antes de atualizar a tabela
        self.tabela.setSortingEnabled(False)
        self.tabela.horizontalHeader().setSortIndicator(-1, Qt.AscendingOrder)

        df_filtrado = self.dados.copy()
        for idx, filtro in enumerate(self.filtros_colunas_widgets):
            texto = filtro.text().strip().lower()
            if texto:
                coluna = self.dados.columns[idx]
                df_filtrado = df_filtrado[df_filtrado[coluna].astype(str).str.lower().str.contains(texto)]

        # 🔧 Reaplicar marcação de duplicados no DataFrame filtrado
        df_filtrado = self.tratar_duplicados(df_filtrado)

        self.tabela.setRowCount(len(df_filtrado))
        self.tabela.setColumnCount(len(df_filtrado.columns))
        self.tabela.setHorizontalHeaderLabels(df_filtrado.columns.tolist())

        for i in range(len(df_filtrado)):
            for j in range(len(df_filtrado.columns)):
                valor = df_filtrado.iat[i, j]
                nome_coluna = df_filtrado.columns[j]

                if nome_coluna in ["Valor Médico", "Valor Convenio"]:
                    try:
                        if str(valor).strip().startswith("R$"):
                            texto = str(valor).strip()
                        else:
                            valor_float = float(str(valor).replace(",", ".").strip())
                            texto = f"R$ {valor_float:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
                    except:
                        texto = str(valor)
                else:
                    texto = str(valor)

                item = QTableWidgetItem(texto)
                item.setTextAlignment(Qt.AlignCenter)

                # 🌿 Verde se exceção OIT
                if "NaoDuplicadoOIT" in df_filtrado.columns and str(df_filtrado.iat[i, df_filtrado.columns.get_loc("NaoDuplicadoOIT")]) == "True":
                    item.setBackground(QtGui.QColor("#C8E6C9"))
                # ⚠️ Amarelo se duplicado
                elif "Duplicado" in df_filtrado.columns and str(df_filtrado.iat[i, df_filtrado.columns.get_loc("Duplicado")]) == "True":
                    item.setBackground(QtGui.QColor("#FFF59D"))

                self.tabela.setItem(i, j, item)

        self.tabela.setUpdatesEnabled(True)
        self.tabela.viewport().update()
        self.tabela.resizeColumnsToContents()
        self.atualizar_resumo(df_filtrado)

        # ✅ Reativa ordenação (limpa e pronta para novo clique, sem bugs)
        self.tabela.setSortingEnabled(True)
        self.tabela.horizontalHeader().setSortIndicator(-1, Qt.AscendingOrder)




    def limpar_filtros_colunas(self):
        # Limpa os campos de filtro
        for filtro in self.filtros_colunas_widgets:
            filtro.blockSignals(True)
            filtro.setText("")
            filtro.blockSignals(False)

        # ⚠️ Remove ordenação do cabeçalho visual
        self.tabela.horizontalHeader().setSortIndicator(-1, Qt.AscendingOrder)

        # 🔁 Recarrega a competência inteira do banco (sem filtros nem ordenação)
        self.aplicar_filtro_competencia()

        # 🔁 Garante que o QTableWidget fique no estado original (sem ordenação visual ativa)
        self.tabela.setSortingEnabled(False)
        self.tabela.horizontalHeader().setSortIndicator(-1, Qt.AscendingOrder)
        self.tabela.setSortingEnabled(True)

        self.label_status.setText("Filtros e ordenações removidos. Competência recarregada.")



    def atualizar_resumo(self, df):
        try:
            total_registros = len(df)
            total_convenios = df["Convenio"].nunique() if "Convenio" in df.columns else 0

            def limpar_valor(valor):
                if isinstance(valor, str):
                    valor = valor.replace("R$", "").replace(".", "").replace(",", ".").strip()
                try:
                    return float(valor)
                except:
                    return 0.0

            total_convenio = df["Valor Convenio"].map(limpar_valor).sum()
            total_medico = df["Valor Médico"].map(limpar_valor).sum()

            total_duplicados = df["Duplicado"].sum() if "Duplicado" in df.columns else 0

            texto = (
                f"Registros: {total_registros} | "
                f"Convênios: {total_convenios} | "
                f"Valor Convênio: R$ {total_convenio:,.2f} | "
                f"Valor Médico: R$ {total_medico:,.2f} | "
                f"Registros Duplicados: {total_duplicados}"
            ).replace(".", "#").replace(",", ".").replace("#", ",")

            self.label_resumo.setText(f"<b>{texto}</b>")

        except Exception as e:
            self.label_resumo.setText(f"Erro ao calcular resumo: {e}")



    def excluir_selecionados(self):
        linhas = list(set(index.row() for index in self.tabela.selectedIndexes()))
        if not linhas:
            QMessageBox.information(self, "Aviso", "Nenhum registro selecionado.")
            return

        confirm = QMessageBox.question(
            self, "Confirmação",
            f"Tem certeza que deseja excluir {len(linhas)} registro(s)?",
            QMessageBox.Yes | QMessageBox.No
        )

        if confirm != QMessageBox.Yes:
            return

        try:
            cods_para_excluir = []
            for linha in linhas:
                item_cod = self.tabela.item(linha, 0)  # Coluna 0 = "Cod Registro"
                if item_cod:
                    cods_para_excluir.append(item_cod.text())

            # Excluir do banco
            conn = sqlite3.connect("db/sistema_financeiro.db")
            cursor = conn.cursor()
            cursor.executemany(
                'DELETE FROM registros_financeiros WHERE "Cod Registro" = ?',
                [(cod,) for cod in cods_para_excluir]
            )
            conn.commit()
            conn.close()

            # 🧹 Limpa filtros e ordenações exatamente como limpar_filtros_colunas
            for filtro in self.filtros_colunas_widgets:
                filtro.blockSignals(True)
                filtro.setText("")
                filtro.blockSignals(False)

            self.tabela.setSortingEnabled(False)
            self.tabela.horizontalHeader().setSortIndicator(-1, Qt.AscendingOrder)
            self.tabela.setSortingEnabled(True)

            self.aplicar_filtro_competencia()
            self.label_status.setText(f"{len(cods_para_excluir)} registro(s) excluído(s). Competência recarregada.")

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao excluir registros:\n{str(e)}")



    def formatar_valores_monetarios(self, df):
        for coluna in ["Valor Médico", "Valor Convenio"]:
            if coluna in df.columns:
                df[coluna] = pd.to_numeric(df[coluna], errors='coerce').fillna(0)
                df[coluna] = df[coluna].astype(float).map(
                    lambda x: f"R$ {x:,.2f}".replace(".", "#").replace(",", ".").replace("#", ",")
                )
        # Garante visualmente que os duplicados tenham "R$ 0,00" no Valor Convênio
        if "Duplicado" in df.columns and "Valor Convenio" in df.columns:
            df.loc[df["Duplicado"] == True, "Valor Convenio"] = "R$ 0,00"
        return df



    def tratar_duplicados(self, df):
        df["Duplicado"] = False
        df["NaoDuplicadoOIT"] = False

        if df.empty or "Nome" not in df.columns or "Procedimento" not in df.columns:
            return df

        colunas_chave = ["Nome", "Empresa", "Convenio", "Tipo de Exame", "Procedimento", "Data Exame"]

        oit_df = df[df["Procedimento"].str.upper().str.contains("RX - TÓRAX OIT - 2 ASSINATURAS", na=False)].copy()
        restantes_df = df[~df.index.isin(oit_df.index)].copy()

        for df_padronizar in [oit_df, restantes_df]:
            if "Empresa" in df_padronizar.columns:
                df_padronizar["Empresa"] = df_padronizar["Empresa"].fillna("").astype(str).str.strip()

        if all(c in oit_df.columns for c in colunas_chave):
            grupo_oit = oit_df.groupby(colunas_chave)

            for _, grupo in grupo_oit:
                indices = grupo.index.tolist()
                medicos = grupo["Médico"].fillna("").astype(str)

                if medicos.nunique() >= 2:
                    oit_df.loc[indices[:2], "NaoDuplicadoOIT"] = True
                    if len(indices) > 2:
                        oit_df.loc[indices[2:], "Duplicado"] = True
                else:
                    if len(indices) > 1:
                        oit_df.loc[indices[1:], "Duplicado"] = True

        if all(c in restantes_df.columns for c in colunas_chave):
            duplicatas = restantes_df.duplicated(subset=colunas_chave, keep='first')
            restantes_df.loc[duplicatas, "Duplicado"] = True

        df_final = pd.concat([oit_df, restantes_df]).sort_index()

        if "Valor Convênio" in df_final.columns:
            df_final.loc[df_final["Duplicado"] == True, "Valor Convênio"] = 0

        return df_final


    
    def exportar_para_excel(self):
        if not hasattr(self, 'dados') or self.dados.empty:
            QMessageBox.information(self, "Aviso", "Nenhum dado carregado para exportar.")
            return

        df_exportar = self.obter_dados_visiveis()
        if df_exportar.empty:
            QMessageBox.information(self, "Aviso", "Nenhum dado visível para exportar.")
            return

        mes = self.combo_mes.currentText()
        ano = self.combo_ano.currentText()
        if mes == "Todos" or ano == "Todos":
            QMessageBox.warning(self, "Atenção", "Selecione uma competência válida para exportar.")
            return

        competencia = f"{mes}/{ano}"
        nome_arquivo = f"Faturamento_Completo_{competencia.replace('/', '-')}.xlsx"
        caminho, _ = QFileDialog.getSaveFileName(self, "Salvar como", nome_arquivo, "Excel Files (*.xlsx)")
        if not caminho:
            return

        try:
            df = self.dados.copy()

            # 🔧 LIMPA formatação monetária antes de tratar duplicados
            for coluna in ["Valor Médico", "Valor Convenio"]:
                if coluna in df.columns:
                    df[coluna] = pd.to_numeric(
                        df[coluna].astype(str)
                        .str.replace("R$", "", regex=False)
                        .str.replace(".", "", regex=False)
                        .str.replace(",", ".", regex=False),
                        errors="coerce"
                    ).fillna(0)

            df = self.tratar_duplicados(df)
            df = self.formatar_valores_monetarios(df)  # <- reaplica R$ e zera duplicados

            if "Médico" in df.columns:
                df["Médico"] = df["Médico"].str.extract(r"^(.*?)(?:\s*-\s*CRM|$)")[0].str.strip()

            colunas_para_remover = ["Competência", "Cod Registro"]
            df.drop(columns=[col for col in colunas_para_remover if col in df.columns], inplace=True)

            colunas_visiveis = [
                col for col in df.columns if col not in ['Duplicado', 'NaoDuplicadoOIT', 'Data Cadastro', 'Data Recebimento', 'Competencia']
            ]

            writer = pd.ExcelWriter(caminho, engine='xlsxwriter')
            workbook = writer.book
            df.to_excel(writer, index=False, sheet_name='Registros', columns=colunas_visiveis, startrow=2)
            worksheet = writer.sheets['Registros']

            # Estilos e cabeçalho
            azul_cabecalho = '#B7D6F4'
            header_format = workbook.add_format({'bold': True, 'bg_color': '#D9D9D9', 'border': 1, 'align': 'center', 'valign': 'vcenter'})
            center_format = workbook.add_format({'align': 'center', 'valign': 'vcenter'})
            money_format = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center'})
            amarelo_center = workbook.add_format({'align': 'center', 'bg_color': '#FFF59D'})
            amarelo_money = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center', 'bg_color': '#FFF59D'})
            verde_center = workbook.add_format({'align': 'center', 'bg_color': '#C8E6C9'})
            verde_money = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center', 'bg_color': '#C8E6C9'})
            bold_blue = workbook.add_format({'bold': True, 'font_color': 'black', 'bg_color': azul_cabecalho, 'border': 1, 'align': 'center'})
            bold_right = workbook.add_format({'bold': True, 'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            value_right = workbook.add_format({'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            money_right = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            logo_background = workbook.add_format({'bg_color': azul_cabecalho, 'border': 1})

            total_laudos = len(df)
            total_valor_medico = df["Valor Médico"].apply(
                lambda v: float(str(v).replace("R$", "").replace(".", "").replace(",", ".")) if isinstance(v, str) else v
            ).sum()
            total_valor_convenio = df["Valor Convenio"].apply(
                lambda v: float(str(v).replace("R$", "").replace(".", "").replace(",", ".")) if isinstance(v, str) else v
            ).sum()

            # Cabeçalho superior
            worksheet.merge_range("A1:A2", "", logo_background)
            worksheet.merge_range("B1:F1", "FATURAMENTO GERAL", bold_blue)
            worksheet.merge_range("B2:F2", f"Competência: {competencia}", bold_blue)
            worksheet.write("G1", "Total de Laudos", bold_right)
            worksheet.write("G2", total_laudos, value_right)
            worksheet.write("H1", "Total Médico", bold_right)
            worksheet.write("H2", total_valor_medico, money_right)
            worksheet.write("I1", "Total Convênio", bold_right)
            worksheet.write("I2", total_valor_convenio, money_right)

            caminho_logo = os.path.join("icones", "medical_laudos_logo.png")
            if os.path.exists(caminho_logo):
                worksheet.insert_image("A1", caminho_logo, {'x_scale': 0.45, 'y_scale': 0.45, 'x_offset': 2, 'y_offset': 2})

            for col_num, nome_coluna in enumerate(colunas_visiveis):
                worksheet.write(2, col_num, nome_coluna, header_format)
                largura_max = max([len(str(cell)) for cell in [nome_coluna] + df[nome_coluna].astype(str).tolist()]) + 2
                worksheet.set_column(col_num, col_num, largura_max)

            worksheet.autofilter(2, 0, 2 + len(df), len(colunas_visiveis) - 1)

            def str_to_bool(valor):
                return str(valor).strip().lower() in ["1", "true", "verdadeiro"]

            for row_num, row_data in df.iterrows():
                duplicado = str_to_bool(row_data.get("Duplicado", ""))
                nao_oit = str_to_bool(row_data.get("NaoDuplicadoOIT", ""))

                for col_num, nome_coluna in enumerate(colunas_visiveis):
                    valor = row_data[nome_coluna]
                    if duplicado:
                        formato = amarelo_money if nome_coluna in ['Valor Médico', 'Valor Convenio'] else amarelo_center
                    elif nao_oit:
                        formato = verde_money if nome_coluna in ['Valor Médico', 'Valor Convenio'] else verde_center
                    else:
                        formato = money_format if nome_coluna in ['Valor Médico', 'Valor Convenio'] else center_format

                    linha_excel = row_num + 3
                    if nome_coluna in ['Valor Médico', 'Valor Convenio']:
                        try:
                            valor_float = float(str(valor).replace("R$", "").replace(".", "").replace(",", "."))
                            worksheet.write_number(linha_excel, col_num, valor_float, formato)
                        except:
                            worksheet.write(linha_excel, col_num, str(valor), formato)
                    else:
                        worksheet.write(linha_excel, col_num, str(valor), formato)

            writer.close()
            QMessageBox.information(self, "Sucesso", f"Arquivo exportado com sucesso para:\n{caminho}")

        except Exception as e:
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self, "Erro", f"Erro ao exportar Excel:\n{e}")




