from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QPushButton, QTableWidget,
    QTableWidgetItem, QLineEdit, QMessageBox, QHeaderView, QFrame
)
from PyQt5.QtCore import Qt
import sqlite3
import os

from gui.dialog_novo_vendedor import DialogNovoVendedor
from gui.dialog_vincular_convenios import DialogVincularConvenios
from gui.dialog_exibir_comissao import DialogExibirComissao

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

class ModuloCadastroVendedores(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.parent_main = parent
        self.layout = QVBoxLayout(self)
        self.layout.setSpacing(20)
        self.layout.setContentsMargins(20, 20, 20, 20)

        estilo_botao = """
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #2980b9;
            }
        """

        # Botões
        botoes_layout = QHBoxLayout()
        self.botao_novo = QPushButton("➕ Novo Vendedor")
        self.botao_editar = QPushButton("✏️ Editar Vendedor")
        self.botao_vincular = QPushButton("🔗 Vincular Convênios")
        self.botao_exibir_comissao = QPushButton("📊 Exibir Comissão")
        self.botao_excluir = QPushButton("🗑️ Excluir Vendedor")
        

        # Aplica o estilo aos botões corretamente
        for botao in [self.botao_novo, self.botao_editar, self.botao_vincular, self.botao_exibir_comissao, self.botao_excluir]:
            botao.setStyleSheet(estilo_botao)


        self.botao_novo.clicked.connect(self.abrir_modal_novo_vendedor)
        self.botao_editar.clicked.connect(self.abrir_modal_editar_vendedor)
        self.botao_excluir.clicked.connect(self.excluir_vendedor)
        self.botao_vincular.clicked.connect(self.abrir_modal_vincular_convenios)
        self.botao_exibir_comissao.clicked.connect(self.abrir_modal_exibir_comissao)


        botoes_layout.addWidget(self.botao_novo)
        botoes_layout.addWidget(self.botao_editar)
        botoes_layout.addWidget(self.botao_vincular)
        botoes_layout.addWidget(self.botao_exibir_comissao)
        botoes_layout.addWidget(self.botao_excluir)

        # Campo de busca
        filtro_layout = QHBoxLayout()
        self.campo_busca = QLineEdit()
        self.campo_busca.setPlaceholderText("🔍 Buscar por nome do vendedor...")
        self.campo_busca.textChanged.connect(self.filtrar_vendedores)
        filtro_layout.addWidget(self.campo_busca)

        # Separador
        linha_divisoria = QFrame()
        linha_divisoria.setFrameShape(QFrame.HLine)
        linha_divisoria.setFrameShadow(QFrame.Sunken)

        # Tabela
        self.tabela = QTableWidget()
        self.tabela.setColumnCount(3)
        self.tabela.setHorizontalHeaderLabels(["Nome", "CPF", "Comissão (%)"])
        self.tabela.verticalHeader().setVisible(False)
        self.tabela.setSortingEnabled(True)
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

        # Layout final
        self.layout.addLayout(botoes_layout)
        self.layout.addLayout(filtro_layout)
        self.layout.addWidget(linha_divisoria)
        self.layout.addWidget(self.tabela)

        self.carregar_vendedores()

    def abrir_modal_novo_vendedor(self):
        dialog = DialogNovoVendedor(self)
        if dialog.exec_() == dialog.Accepted:
            self.carregar_vendedores()

    def carregar_vendedores(self):
        self.tabela.setRowCount(0)
        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("""
                CREATE TABLE IF NOT EXISTS vendedores (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    nome TEXT NOT NULL,
                    cpf TEXT NOT NULL,
                    comissao_padrao REAL NOT NULL
                )
            """)
            cursor.execute("SELECT nome, cpf, comissao_padrao FROM vendedores ORDER BY nome")
            vendedores = cursor.fetchall()
            conn.close()

            self.tabela.setRowCount(len(vendedores))
            for i, (nome, cpf, comissao) in enumerate(vendedores):
                item_nome = QTableWidgetItem(nome)
                item_nome.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 0, item_nome)

                item_cpf = QTableWidgetItem(cpf)
                item_cpf.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 1, item_cpf)

                item_comissao = QTableWidgetItem(f"{comissao:.2f}")
                item_comissao.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 2, item_comissao)

        except Exception as e:
            print(f"[ERRO] ao carregar vendedores: {e}")

    def excluir_vendedor(self):
        row = self.tabela.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Aviso", "Selecione um vendedor para excluir.")
            return

        nome = self.tabela.item(row, 0).text()
        cpf = self.tabela.item(row, 1).text()

        confirmar = QMessageBox.question(
            self,
            "Confirmar exclusão",
            f"Deseja excluir o vendedor '{nome}' ({cpf})?",
            QMessageBox.Yes | QMessageBox.No
        )
        if confirmar == QMessageBox.Yes:
            try:
                conn = sqlite3.connect(CAMINHO_BANCO)
                cursor = conn.cursor()
                cursor.execute("DELETE FROM vendedores WHERE nome = ? AND cpf = ?", (nome, cpf))
                conn.commit()
                conn.close()
                self.carregar_vendedores()
                QMessageBox.information(self, "Sucesso", f"Vendedor '{nome}' excluído.")
            except Exception as e:
                QMessageBox.critical(self, "Erro", f"Erro ao excluir vendedor:\n{e}")

    def filtrar_vendedores(self):
        texto = self.campo_busca.text().lower()
        for row in range(self.tabela.rowCount()):
            item_nome = self.tabela.item(row, 0)
            if item_nome and texto in item_nome.text().lower():
                self.tabela.setRowHidden(row, False)
            else:
                self.tabela.setRowHidden(row, True)

    def abrir_modal_vincular_convenios(self):
        row = self.tabela.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Aviso", "Selecione um vendedor.")
            return

        nome = self.tabela.item(row, 0).text()
        dialog = DialogVincularConvenios(nome, self)
        dialog.exec_()

    def abrir_modal_exibir_comissao(self):
        row = self.tabela.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Aviso", "Selecione um vendedor.")
            return

        nome = self.tabela.item(row, 0).text()
        dialog = DialogExibirComissao(nome, self)
        dialog.exec_()

    def abrir_modal_editar_vendedor(self):
        row = self.tabela.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Aviso", "Selecione um vendedor para editar.")
            return

        nome = self.tabela.item(row, 0).text()
        cpf = self.tabela.item(row, 1).text()
        comissao = float(self.tabela.item(row, 2).text())

        dados_existentes = {
            "nome": nome,
            "cpf": cpf,
            "comissao": comissao
        }

        dialog = DialogNovoVendedor(self, dados_existentes)
        if dialog.exec_() == dialog.Accepted:
            self.carregar_vendedores()



