from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QLabel, QPushButton, QComboBox,
    QTableWidget, QTableWidgetItem, QHeaderView, QLineEdit, QFrame, QMessageBox,
    QAbstractItemView
)
from PyQt5.QtCore import Qt
import sqlite3, os
from datetime import datetime

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")


def ensure_schema_comissoes():
    """Cria a tabela agregada de comissões por vendedor/competência (se não existir)."""
    with sqlite3.connect(CAMINHO_BANCO) as con:
        cur = con.cursor()
        cur.execute(
            """
            CREATE TABLE IF NOT EXISTS comissoes_vendedores (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                vendedor_id INTEGER NOT NULL,
                competencia TEXT NOT NULL,
                total_base REAL DEFAULT 0,
                total_comissao REAL DEFAULT 0,
                registros INTEGER DEFAULT 0,
                pago INTEGER DEFAULT 0,
                data_calculo TEXT,
                UNIQUE(vendedor_id, competencia)
            )
            """
        )
        cur.execute("CREATE INDEX IF NOT EXISTS idx_comissoes_comp ON comissoes_vendedores(competencia)")
        con.commit()


def parse_money(v):
    try:
        if isinstance(v, (int, float)):
            return float(v)
        s = str(v or "").replace("R$", "").replace(".", "").replace(",", ".").strip()
        return float(s) if s else 0.0
    except Exception:
        return 0.0


def fmt_money(n):
    s = f"{float(n):,.2f}"
    s = s.replace(",", "X").replace(".", ",").replace("X", ".")
    return f"R$ {s}"


class ModuloInfoVendedores(QWidget):
    """Aba 'Informações' — calcula/mostra comissões por competência e controla pagamento."""
    def __init__(self, parent=None):
        super().__init__(parent)
        ensure_schema_comissoes()

        root = QVBoxLayout(self)
        root.setContentsMargins(20, 20, 20, 20)
        root.setSpacing(20)

        estilo_botao = (
            """            QPushButton {
                background-color: #3498db; color: white; padding: 6px 12px;
                font-weight: bold; border-radius: 4px;
            }
            QPushButton:hover { background-color: #2980b9; }
            """
        )

        # ===== Topbar: competência + ações =====
        top = QHBoxLayout(); top.setSpacing(8)

        top.addWidget(QLabel("Competência:"))
        self.combo_comp = QComboBox(); self.combo_comp.setMinimumWidth(130)
        top.addWidget(self.combo_comp)

        self.btn_recalcular = QPushButton("🔄 Recalcular")
        self.btn_detalhar   = QPushButton("📑 Detalhar Selecionado")
        self.btn_pago       = QPushButton("💰 Marcar Pago")
        self.btn_aberto     = QPushButton("⛔ Marcar Em Aberto")
        for b in (self.btn_recalcular, self.btn_detalhar, self.btn_pago, self.btn_aberto):
            b.setStyleSheet(estilo_botao)
        top.addStretch(1)
        top.addWidget(self.btn_recalcular)
        top.addWidget(self.btn_detalhar)
        top.addWidget(self.btn_pago)
        top.addWidget(self.btn_aberto)

        root.addLayout(top)

        # ===== Filtro texto =====
        filtro = QHBoxLayout()
        self.campo_busca = QLineEdit(); self.campo_busca.setPlaceholderText("🔍 Buscar por vendedor/CPF/situação…")
        self.campo_busca.textChanged.connect(self._on_busca)
        filtro.addWidget(self.campo_busca)
        root.addLayout(filtro)

        # ===== Separador =====
        sep = QFrame(); sep.setFrameShape(QFrame.HLine); sep.setFrameShadow(QFrame.Sunken)
        root.addWidget(sep)

        # ===== Tabela =====
        self.tabela = QTableWidget()
        self.tabela.setColumnCount(7)
        self.tabela.setHorizontalHeaderLabels([
            "Vendedor", "CPF", "% Comissão", "Registros", "Base (R$)", "Comissão (R$)", "Situação"
        ])
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.tabela.setAlternatingRowColors(True)
        self.tabela.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.tabela.setSelectionMode(QAbstractItemView.SingleSelection)
        self.tabela.verticalHeader().setVisible(False)
        self.tabela.setSortingEnabled(True)
        root.addWidget(self.tabela, 1)

        # ===== Rodapé simples =====
        self.lbl_totais = QLabel("Total comissão: R$ 0,00 | Vendedores pagos: 0 | Em aberto: 0")
        root.addWidget(self.lbl_totais)

        # Conexões
        self.combo_comp.currentIndexChanged.connect(self.carregar_dados)
        self.btn_recalcular.clicked.connect(self.recalcular_competencia)
        self.btn_detalhar.clicked.connect(self.abrir_detalhe)
        self.btn_pago.clicked.connect(lambda: self._set_status(1))
        self.btn_aberto.clicked.connect(lambda: self._set_status(0))

        # Inicialização
        self._carregar_competencias()
        self.carregar_dados()

    # ---------- Banco/consulta ----------
    def _carregar_competencias(self):
        self.combo_comp.clear()
        try:
            with sqlite3.connect(CAMINHO_BANCO) as con:
                cur = con.cursor()
                cur.execute("SELECT DISTINCT Competência FROM registros_financeiros WHERE Competência IS NOT NULL ORDER BY Competência DESC")
                comps = [r[0] for r in cur.fetchall() if r[0]]
            self.combo_comp.addItems(comps)
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar competências:\n{e}")

    def _competencia(self):
        return self.combo_comp.currentText() or ""

    def recalcular_competencia(self):
        comp = self._competencia()
        if not comp:
            QMessageBox.warning(self, "Aviso", "Selecione uma competência.")
            return

        try:
            with sqlite3.connect(CAMINHO_BANCO) as con:
                cur = con.cursor()

                # Vendedores com comissão e CPF
                cur.execute("SELECT id, nome, cpf, comissao_padrao FROM vendedores ORDER BY nome")
                vendedores = cur.fetchall()

                for vendedor_id, nome, cpf, comissao in vendedores:
                    # Convênios vinculados
                    cur.execute("SELECT nome_convenio FROM vendedor_convenio WHERE id_vendedor=?", (vendedor_id,))
                    convs = [r[0] for r in cur.fetchall()]

                    # Base e registros
                    base_total = 0.0
                    regs = 0
                    if convs:
                        marks = ",".join(["?"] * len(convs))
                        q = f"""                            SELECT `Valor Convenio` FROM registros_financeiros
                            WHERE Competência=? AND Convenio IN ({marks})
                        """
                        cur.execute(q, [comp] + convs)
                        valores = [parse_money(v[0]) for v in cur.fetchall()]
                        regs = len(valores)
                        base_total = sum(valores)

                    comissao_total = round(base_total * float(comissao or 0) / 100.0, 2)

                    # Preserva status de pagamento anterior (se já existir)
                    cur.execute(
                        "SELECT pago FROM comissoes_vendedores WHERE vendedor_id=? AND competencia=?",
                        (vendedor_id, comp),
                    )
                    row = cur.fetchone()
                    pago_ant = row[0] if row is not None else 0

                    cur.execute(
                        """                        INSERT INTO comissoes_vendedores (vendedor_id, competencia, total_base, total_comissao, registros, pago, data_calculo)
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                        ON CONFLICT(vendedor_id, competencia) DO UPDATE SET
                            total_base=excluded.total_base,
                            total_comissao=excluded.total_comissao,
                            registros=excluded.registros,
                            data_calculo=excluded.data_calculo,
                            pago=COALESCE(comissoes_vendedores.pago, excluded.pago)
                        """,
                        (
                            vendedor_id, comp, base_total, comissao_total, regs, pago_ant,
                            datetime.now().isoformat(timespec="seconds"),
                        ),
                    )

                con.commit()

            self.carregar_dados()
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao recalcular comissão:\n{e}")

    def carregar_dados(self):
        comp = self._competencia()
        self.tabela.setRowCount(0)
        total, pagos, abertos = 0.0, 0, 0

        try:
            with sqlite3.connect(CAMINHO_BANCO) as con:
                cur = con.cursor()
                cur.execute(
                    """                    SELECT c.vendedor_id, v.nome, v.cpf, v.comissao_padrao,
                           c.registros, c.total_base, c.total_comissao, c.pago
                    FROM comissoes_vendedores c
                    JOIN vendedores v ON v.id = c.vendedor_id
                    WHERE c.competencia=?
                    ORDER BY v.nome
                    """,
                    (comp,),
                )
                rows = cur.fetchall()

            for i, (vendedor_id, nome, cpf, pct, regs, base, comissao, pago) in enumerate(rows):
                self.tabela.insertRow(i)

                it_nome = QTableWidgetItem(nome); it_nome.setData(Qt.UserRole, vendedor_id); it_nome.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 0, it_nome)

                it_cpf = QTableWidgetItem(cpf or ""); it_cpf.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 1, it_cpf)

                it_pct = QTableWidgetItem(f"{float(pct or 0):.2f}"); it_pct.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 2, it_pct)

                it_regs = QTableWidgetItem(str(int(regs or 0))); it_regs.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 3, it_regs)

                it_base = QTableWidgetItem(fmt_money(base or 0)); it_base.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 4, it_base)

                it_com = QTableWidgetItem(fmt_money(comissao or 0)); it_com.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 5, it_com)

                sit = "🟢 Pago" if int(pago or 0) == 1 else "⚪ Em aberto"
                it_sit = QTableWidgetItem(sit); it_sit.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, 6, it_sit)

                total += float(comissao or 0)
                if int(pago or 0) == 1:
                    pagos += 1
                else:
                    abertos += 1

            self.lbl_totais.setText(
                f"Total comissão: {fmt_money(total)} | Vendedores pagos: {pagos} | Em aberto: {abertos}"
            )
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar dados:\n{e}")

    # ---------- Ações ----------
    def _sel_vendedor_id(self):
        row = self.tabela.currentRow()
        if row < 0:
            return None
        it = self.tabela.item(row, 0)
        return it.data(Qt.UserRole) if it else None

    def _set_status(self, pago: int):
        vendedor_id = self._sel_vendedor_id()
        if vendedor_id is None:
            QMessageBox.warning(self, "Aviso", "Selecione um vendedor na tabela.")
            return
        comp = self._competencia()
        try:
            with sqlite3.connect(CAMINHO_BANCO) as con:
                cur = con.cursor()
                cur.execute(
                    "UPDATE comissoes_vendedores SET pago=? WHERE vendedor_id=? AND competencia=?",
                    (1 if pago else 0, vendedor_id, comp),
                )
                con.commit()
            self.carregar_dados()
        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao atualizar status:\n{e}")

    def abrir_detalhe(self):
        try:
            from gui.dialog_exibir_comissao import DialogExibirComissao
        except Exception:
            QMessageBox.warning(self, "Aviso", "Janela de detalhamento não encontrada.")
            return

        row = self.tabela.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Aviso", "Selecione um vendedor na tabela.")
            return
        nome = self.tabela.item(row, 0).text()

        dlg = DialogExibirComissao(nome, self)
        dlg.exec_()

    # ---------- UI helpers ----------
    def _on_busca(self):
        texto = (self.campo_busca.text() or "").lower().strip()
        termos = [t for t in texto.split() if t]
        for r in range(self.tabela.rowCount()):
            blob = " ".join([
                (self.tabela.item(r, c).text().lower() if self.tabela.item(r, c) else "")
                for c in range(self.tabela.columnCount())
            ])
            visivel = all(t in blob for t in termos)
            self.tabela.setRowHidden(r, not visivel)
