# modulo6_faturamento_medicos.py

from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QLabel, QPushButton,
    QComboBox, QTableWidget, QTableWidgetItem, QMessageBox, QDialog,
    QFormLayout, QFileDialog, QSizePolicy, QSpacerItem, QHeaderView, QLineEdit
)
from PyQt5.QtCore import Qt
import sqlite3
import pandas as pd
import os
import unicodedata
import re
from PyQt5.QtGui import QFont


CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")


def remover_acentos(texto):
    if not isinstance(texto, str):
        return ""
    return ''.join(c for c in unicodedata.normalize('NFD', texto) if unicodedata.category(c) != 'Mn')


class DialogFaturamentoMedico(QDialog):
    def __init__(self, medico_nome, competencia, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Faturamento do Médico")
        self.setMinimumWidth(500)
        self.medico_nome = medico_nome
        self.competencia = competencia

        layout = QVBoxLayout(self)

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()

            cursor.execute("SELECT * FROM medicos WHERE nome = ?", (medico_nome,))
            dados_medico = cursor.fetchone()
            colunas = [desc[0] for desc in cursor.description]

            if dados_medico:
                form_layout = QFormLayout()
                for i, valor in enumerate(dados_medico):
                    form_layout.addRow(colunas[i].capitalize(), QLabel(str(valor)))
                layout.addLayout(form_layout)

            self.df = pd.read_sql_query(
                "SELECT * FROM registros_financeiros WHERE Médico = ? AND Competência = ?",
                conn, params=(medico_nome, competencia)
            )
            conn.close()

            qtd = len(self.df)
            total = pd.to_numeric(self.df['Valor Médico'], errors='coerce').fillna(0).sum()

            layout.addWidget(QLabel(f"\nQuantidade de exames: {qtd}"))
            layout.addWidget(QLabel(f"Valor total a pagar: R$ {total:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")))

            btn_exportar = QPushButton("Exportar Excel")
            btn_exportar.setStyleSheet("""
                QPushButton {
                    background-color: #3498db;
                    color: white;
                    padding: 6px 12px;
                    font-weight: bold;
                    border-radius: 4px;
                }
                QPushButton:hover {
                    background-color: #2980b9;
                }
            """)
            layout.addWidget(btn_exportar)
            btn_exportar.clicked.connect(self.exportar_excel)

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar faturamento: {e}")

    def exportar_excel(self):

        nome = self.medico_nome
        nome_somente = re.sub(r"\s*-\s*CRM.*", "", nome).strip()
        nome_maiusculo = nome_somente.upper()
        competencia = self.competencia

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            df = pd.read_sql_query(
                "SELECT * FROM registros_financeiros WHERE \"Médico\" = ? AND \"Competência\" = ?",
                conn, params=(nome, competencia)
            )
            conn.close()

            if df.empty:
                QMessageBox.information(self, "Aviso", f"Nenhum registro encontrado para {nome} na competência {competencia}.")
                return

            if "Médico" in df.columns:
                df["Médico"] = df["Médico"].str.extract(r"^(.*?)(?:\\s*-\\s*CRM|$)")[0].str.strip()

            colunas_para_remover = ["Empresa", "Convenio", "Valor Convenio"]
            df.drop(columns=[col for col in colunas_para_remover if col in df.columns], inplace=True)

            caminho, _ = QFileDialog.getSaveFileName(
                self, "Salvar como", f"{nome_somente}_{competencia.replace('/', '-')}.xlsx", "Excel Files (*.xlsx)"
            )
            if not caminho:
                return

            colunas_visiveis = [
                col for col in df.columns if col not in [
                    'Cod Registro', 'Duplicado', 'NaoDuplicadoOIT',
                    'Competência', 'Data Cadastro', 'Data Recebimento',
                    'Tipo de Exame', 'Competencia'
                ]
            ]

            writer = pd.ExcelWriter(caminho, engine='xlsxwriter')
            workbook = writer.book
            df.to_excel(writer, index=False, sheet_name='Registros', columns=colunas_visiveis, startrow=2)
            worksheet = writer.sheets['Registros']

            azul_cabecalho = '#B7D6F4'
            header_format = workbook.add_format({'bold': True, 'bg_color': '#D9D9D9', 'border': 1, 'align': 'center', 'valign': 'vcenter'})
            center_format = workbook.add_format({'align': 'center', 'valign': 'vcenter'})
            money_format = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center', 'valign': 'vcenter'})
            amarelo_center = workbook.add_format({'align': 'center', 'bg_color': '#FFF59D', 'valign': 'vcenter'})
            amarelo_money = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center', 'bg_color': '#FFF59D', 'valign': 'vcenter'})
            verde_center = workbook.add_format({'align': 'center', 'bg_color': '#C8E6C9', 'valign': 'vcenter'})
            verde_money = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'center', 'bg_color': '#C8E6C9', 'valign': 'vcenter'})
            bold_blue = workbook.add_format({'bold': True, 'font_color': 'black', 'bg_color': azul_cabecalho, 'border': 1, 'align': 'center'})
            bold_right = workbook.add_format({'bold': True, 'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            value_right = workbook.add_format({'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            money_right = workbook.add_format({'num_format': '"R$" #,##0.00', 'align': 'right', 'bg_color': azul_cabecalho, 'border': 1})
            logo_background = workbook.add_format({'bg_color': azul_cabecalho, 'border': 1})

            total_laudos = len(df)
            total_valor = df["Valor Médico"].apply(
                lambda v: float(str(v).replace("R$", "").replace(".", "").replace(",", ".")) if isinstance(v, str) else v
            ).sum()

            worksheet.merge_range("A1:A2", "", logo_background)
            worksheet.merge_range("B1:C1", f"FATURAMENTO {nome_maiusculo}", bold_blue)
            worksheet.merge_range("B2:C2", f"COMPETÊNCIA: {competencia}", bold_blue)
            worksheet.write("D1", "TOTAL", bold_right)
            worksheet.write("E1", total_valor, money_right)
            worksheet.write("D2", "LAUDOS", bold_right)
            worksheet.write("E2", total_laudos, value_right)

            caminho_logo = os.path.join("icones", "medical_laudos_logo.png")
            if os.path.exists(caminho_logo):
                worksheet.insert_image("A1", caminho_logo, {'x_scale': 0.45, 'y_scale': 0.45, 'x_offset': 2, 'y_offset': 2})

            for col_num, nome_coluna in enumerate(colunas_visiveis):
                worksheet.write(2, col_num, nome_coluna, header_format)
                largura = max(df[nome_coluna].astype(str).map(len).max(), len(nome_coluna)) + 2
                worksheet.autofilter(2, 0, 2 + len(df), len(colunas_visiveis) - 1)
                worksheet.set_column(col_num, col_num, largura)

            def str_to_bool(valor):
                return str(valor).strip().lower() in ["1", "true", "verdadeiro"]

            for row_num, row_data in df.iterrows():
                duplicado = str_to_bool(row_data.get("Duplicado", ""))
                nao_oit = str_to_bool(row_data.get("NaoDuplicadoOIT", ""))

                for col_num, nome_coluna in enumerate(colunas_visiveis):
                    valor = row_data[nome_coluna]
                    if duplicado:
                        formato = amarelo_money if nome_coluna == 'Valor Médico' else amarelo_center
                    elif nao_oit:
                        formato = verde_money if nome_coluna == 'Valor Médico' else verde_center
                    else:
                        formato = money_format if nome_coluna == 'Valor Médico' else center_format

                    linha_excel = row_num + 3
                    if nome_coluna == 'Valor Médico':
                        try:
                            valor_float = float(valor)
                            worksheet.write_number(linha_excel, col_num, valor_float, formato)
                        except:
                            worksheet.write(linha_excel, col_num, str(valor), formato)
                    else:
                        worksheet.write(linha_excel, col_num, str(valor), formato)

            writer.close()
            QMessageBox.information(self, "Sucesso", f"Arquivo exportado com sucesso para:\n{caminho}")

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao exportar Excel:\n{e}")




class ModuloFaturamentoMedicos(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.layout = QVBoxLayout(self)

        titulo = QLabel("Faturamento de Médicos")
        titulo.setAlignment(Qt.AlignCenter)
        titulo.setStyleSheet("font-size: 20px; font-weight: bold; margin-bottom: 15px;")
        self.layout.addWidget(titulo)

        self.combo_mes = QComboBox()
        self.combo_ano = QComboBox()
        self.botao_carregar = QPushButton("Carregar Médicos")

        self.botao_carregar.setStyleSheet("""
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #2980b9;
            }
        """)


        self.combo_mes.addItems([f"{i:02d}" for i in range(1, 13)])
        self.combo_ano.addItems([str(ano) for ano in range(2025, 2035)])

        seletor_layout = QHBoxLayout()
        seletor_layout.setSpacing(10)
        seletor_layout.addWidget(QLabel("Mês:"))
        seletor_layout.addWidget(self.combo_mes)
        seletor_layout.addWidget(QLabel("Ano:"))
        seletor_layout.addWidget(self.combo_ano)
        seletor_layout.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        seletor_layout.addWidget(self.botao_carregar)

        self.layout.addLayout(seletor_layout)

        self.input_busca = QLineEdit()
        self.input_busca.setPlaceholderText("🔍 Buscar por nome do médico...")
        self.input_busca.textChanged.connect(self.aplicar_filtro_busca)
        self.layout.addWidget(self.input_busca)

        self.tabela = QTableWidget()
        self.tabela.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        self.tabela.setAlternatingRowColors(True)
        self.tabela.setSelectionBehavior(QTableWidget.SelectRows)
        self.tabela.setSelectionMode(QTableWidget.SingleSelection)
        self.tabela.horizontalHeader().setStretchLastSection(True)
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        self.layout.addWidget(self.tabela)

        rodape = QHBoxLayout()
        self.label_totais = QLabel("Médicos: 0 | Total de exames: 0 | Total a pagar: R$ 0,00 | ✅ Total Pago: R$ 0,00 | ❌ Total em Aberto: R$ 0,00")
        self.label_totais.setStyleSheet("font-weight: bold; padding: 6px 12px; color: #2c3e50;")
        self.botao_exibir = QPushButton("Exibir Faturamento do Médico")
        self.botao_exibir.setEnabled(False)
        self.botao_exibir.setStyleSheet("""
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #2980b9;
            }
        """)

        self.botao_alterar_pagamento = QPushButton("Alterar Pagamento")
        self.botao_alterar_pagamento.setEnabled(False)
        self.botao_alterar_pagamento.setStyleSheet("""
            QPushButton {
                background-color: #27ae60;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #219150;
            }
        """)
        rodape.addWidget(self.botao_alterar_pagamento)

        self.botao_alterar_pagamento.clicked.connect(self.alterar_pagamento)


        rodape.addWidget(self.label_totais)
        rodape.addSpacerItem(QSpacerItem(40, 20, QSizePolicy.Expanding, QSizePolicy.Minimum))
        rodape.addWidget(self.botao_exibir)
        self.layout.addLayout(rodape)

        self.botao_carregar.clicked.connect(self.carregar_medicos)
        self.botao_exibir.clicked.connect(self.exibir_modal)
        self.tabela.cellClicked.connect(lambda: self.botao_exibir.setEnabled(True))
        self.tabela.cellClicked.connect(lambda: self.botao_exibir.setEnabled(True) or self.botao_alterar_pagamento.setEnabled(True))


    def carregar_medicos(self):
        self.botao_exibir.setEnabled(False)
        mes = self.combo_mes.currentText()
        ano = self.combo_ano.currentText()
        self.competencia = f"{mes}/{ano}"

        try:
            conn = sqlite3.connect(CAMINHO_BANCO)

            df = pd.read_sql_query(
                "SELECT * FROM registros_financeiros WHERE Competência = ?",
                conn, params=(self.competencia,)
            )
            df["Valor Médico"] = pd.to_numeric(df["Valor Médico"], errors='coerce').fillna(0)
            df_grouped = df.groupby("Médico").agg(
                Quantidade=("Médico", "count"),
                Total=("Valor Médico", "sum")
            ).reset_index()

            # 🧾 Consulta de pagamentos (sem fechar ainda!)
            pagamentos = pd.read_sql_query("""
                SELECT nome_medico, competencia, valor_pago FROM pagamentos_medicos_competencia
                WHERE competencia = ?
            """, conn, params=(self.competencia,))

            conn.close()  # ✅ agora sim pode fechar!

            # Junta com os dados agrupados
            df_grouped = df_grouped.merge(
                pagamentos, how="left", left_on="Médico", right_on="nome_medico"
            )
            df_grouped.drop(columns=["nome_medico"], inplace=True)
            df_grouped["valor_pago"] = df_grouped["valor_pago"].fillna("Não pago")

            self.df_completo = df_grouped
            self.atualizar_tabela(df_grouped)

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao carregar dados: {e}")


    def aplicar_filtro_busca(self):
        texto = remover_acentos(self.input_busca.text().strip().lower())
        df_filtrado = self.df_completo.copy()

        if texto:
            df_filtrado["Médico_normalizado"] = df_filtrado["Médico"].astype(str).apply(lambda x: remover_acentos(x).strip().lower())
            df_filtrado = df_filtrado[df_filtrado["Médico_normalizado"].str.contains(texto, na=False)]
            df_filtrado.drop(columns=["Médico_normalizado"], inplace=True)

        self.atualizar_tabela(df_filtrado)

    def atualizar_tabela(self, df):
        self.tabela.clear()
        self.tabela.setRowCount(len(df))
        self.tabela.setColumnCount(4)
        self.tabela.setHorizontalHeaderLabels(["Médico", "Quantidade", "Total a Pagar", "Pagamento"])


        for i in range(len(df)):
            item_medico = QTableWidgetItem(str(df.iloc[i]["Médico"]))
            item_medico.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(i, 0, item_medico)

            item_qtd = QTableWidgetItem(str(df.iloc[i]["Quantidade"]))
            item_qtd.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(i, 1, item_qtd)

            total_formatado = f"R$ {df.iloc[i]['Total']:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
            item_total = QTableWidgetItem(total_formatado)
            item_total.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(i, 2, item_total)

            # Status de pagamento
            valor = df.iloc[i]["valor_pago"]
            texto_status = "✅ Pago" if valor == "Pago" else "❌ Não pago"

            item_pagamento = QTableWidgetItem(texto_status)
            item_pagamento.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(i, 3, item_pagamento)




        total_medicos = len(df)
        total_exames = df["Quantidade"].sum()
        total_geral = df["Total"].sum()

        df["PagoBool"] = df["valor_pago"].str.strip().str.lower() == "pago"
        total_pago = df[df["PagoBool"]]["Total"].sum()
        total_aberto = total_geral - total_pago

        # Formatando os valores
        fmt = lambda v: f"R$ {v:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        total_geral_str = fmt(total_geral)
        total_pago_str = fmt(total_pago)
        total_aberto_str = fmt(total_aberto)

        self.label_totais.setText(
            f"Médicos: {total_medicos} | Total de exames: {total_exames} | Total a pagar: {total_geral_str} | ✅ Total Pago: {total_pago_str} | ❌ Total em Aberto: {total_aberto_str}"
)


    def exibir_modal(self):
        selected = self.tabela.currentRow()
        if selected < 0:
            QMessageBox.warning(self, "Atenção", "Selecione um médico na tabela.")
            return

        nome = self.tabela.item(selected, 0).text()
        dialog = DialogFaturamentoMedico(nome, self.competencia)
        dialog.exec_()


    def alterar_pagamento(self):
        row = self.tabela.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Aviso", "Selecione um médico.")
            return
        

        nome = self.tabela.item(row, 0).text()
        status_atual = self.tabela.item(row, 3).text().strip().lower()

        if "pago" in status_atual and "não" not in status_atual:
            novo_status = "Não pago"
        else:
            novo_status = "Pago"


        try:
            conn = sqlite3.connect(CAMINHO_BANCO)
            cursor = conn.cursor()
            cursor.execute("""
                INSERT INTO pagamentos_medicos_competencia (nome_medico, competencia, valor_pago)
                VALUES (?, ?, ?)
                ON CONFLICT(nome_medico, competencia) DO UPDATE SET valor_pago = excluded.valor_pago
            """, (nome, self.competencia, novo_status))
            conn.commit()
            conn.close()

            QMessageBox.information(self, "Sucesso", f"Pagamento alterado para: {novo_status}")
            self.carregar_medicos()

        except Exception as e:
            QMessageBox.critical(self, "Erro", f"Erro ao alterar pagamento:\n{e}")
