from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QPushButton, QTableWidget,
    QTableWidgetItem, QDialog, QFormLayout, QLineEdit, QComboBox,
    QMessageBox, QLabel, QDateEdit, QTabWidget, QHeaderView
)
from PyQt5.QtCore import Qt, QDate
import sqlite3
import os
from datetime import datetime
from PyQt5.QtGui import QDoubleValidator

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

# ----------------------------
# Aba 1 - Cadastro de Prestadores
# ----------------------------

class DialogPrestador(QDialog):
    def __init__(self, prestador=None, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Prestador de Serviço")
        self.setFixedSize(350, 250)
        layout = QFormLayout()

        self.input_nome = QLineEdit()
        self.input_documento = QLineEdit()
        self.input_servico = QLineEdit()
        self.input_valor = QLineEdit()
        validator = QDoubleValidator(0.00, 999999.99, 2)
        validator.setNotation(QDoubleValidator.StandardNotation)
        self.input_valor.setValidator(validator)

        self.combo_status = QComboBox()
        self.combo_status.addItems(["Ativo", "Inativo"])

        layout.addRow("Nome:", self.input_nome)
        layout.addRow("CPF/CNPJ:", self.input_documento)
        layout.addRow("Tipo de Serviço:", self.input_servico)
        layout.addRow("Valor Mensal (R$):", self.input_valor)
        layout.addRow("Status:", self.combo_status)

        self.btn_salvar = QPushButton("Salvar")
        self.btn_salvar.clicked.connect(self.accept)
        layout.addRow(self.btn_salvar)
        self.setLayout(layout)

        if prestador:
            self.input_nome.setText(prestador[1])
            self.input_documento.setText(prestador[2])
            self.input_servico.setText(prestador[3])
            self.input_valor.setText(str(prestador[4]))
            self.combo_status.setCurrentText(prestador[5])

    def get_data(self):
        return (
            self.input_nome.text().strip(),
            self.input_documento.text().strip(),
            self.input_servico.text().strip(),
            float(self.input_valor.text().replace(",", ".") or 0),
            self.combo_status.currentText()
        )

class AbaPrestadores(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Cadastro de Prestadores")
        self.layout = QVBoxLayout(self)

        # Botões
        botoes_layout = QHBoxLayout()
        self.btn_novo = QPushButton("+ Criar Prestador")
        self.btn_editar = QPushButton("Editar")
        self.btn_excluir = QPushButton("Excluir")
        self.btn_editar.setEnabled(False)
        self.btn_excluir.setEnabled(False)

        botoes_layout.addWidget(self.btn_novo)
        botoes_layout.addWidget(self.btn_editar)
        botoes_layout.addWidget(self.btn_excluir)
        self.layout.addLayout(botoes_layout)

        # Campo de filtro
        filtro_layout = QHBoxLayout()
        filtro_layout.addWidget(QLabel("Filtrar por nome:"))
        self.input_filtro = QLineEdit()
        self.input_filtro.setPlaceholderText("🔍 Digite o nome do prestador...")
        self.input_filtro.textChanged.connect(self.carregar_dados)
        filtro_layout.addWidget(self.input_filtro)
        self.layout.addLayout(filtro_layout)


        # Tabela
        self.tabela = QTableWidget()
        self.tabela.setColumnCount(5)
        self.tabela.setHorizontalHeaderLabels(["Nome", "CPF/CNPJ", "Serviço", "Valor (R$)", "Status"])
        self.tabela.setSelectionBehavior(QTableWidget.SelectRows)

        # 🛠 Ajuste para pegar a tela toda
        self.tabela.horizontalHeader().setStretchLastSection(True)
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

        self.layout.addWidget(self.tabela)

        # Sinais
        self.tabela.itemSelectionChanged.connect(self.atualizar_botoes)
        self.btn_novo.clicked.connect(self.criar_prestador)
        self.btn_editar.clicked.connect(self.editar_prestador)
        self.btn_excluir.clicked.connect(self.excluir_prestador)

        self.label_rodape = QLabel("Total de prestadores: 0 | Valor total: R$ 0,00")
        self.layout.addWidget(self.label_rodape)


        self.carregar_dados()

    def conexao(self):
        return sqlite3.connect(CAMINHO_BANCO)

    def carregar_dados(self):
        self.tabela.setRowCount(0)
        conn = self.conexao()
        cursor = conn.cursor()
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS prestadores (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                nome TEXT,
                cpf_cnpj TEXT,
                tipo_servico TEXT,
                valor_mensal REAL,
                status TEXT DEFAULT 'Ativo'
            )
        """)
        filtro_nome = self.input_filtro.text().strip().lower()

        if filtro_nome:
            cursor.execute("""
                SELECT * FROM prestadores
                WHERE lower(nome) LIKE ?
                ORDER BY nome
            """, (f"%{filtro_nome}%",))
        else:
            cursor.execute("SELECT * FROM prestadores ORDER BY nome")

        prestadores = cursor.fetchall()

        conn.close()

        total_valor = 0
        for row_idx, row in enumerate(prestadores):
            self.tabela.insertRow(row_idx)
            for col_idx, valor in enumerate(row[1:]):
                item = QTableWidgetItem(str(valor))
                item.setFlags(Qt.ItemIsSelectable | Qt.ItemIsEnabled)
                item.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(row_idx, col_idx, item)

            # soma apenas valores numéricos (coluna 4: valor_mensal)
            if isinstance(row[4], (int, float)):
                total_valor += row[4]

        # Atualiza rodapé
        total_prestadores = len(prestadores)
        valor_formatado = f"R$ {total_valor:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        self.label_rodape.setText(
            f"<b>Total de prestadores:</b> {total_prestadores} &nbsp;&nbsp;|&nbsp;&nbsp; <b>Valor total:</b> {valor_formatado}"
        )





    def atualizar_botoes(self):
        selecionado = self.tabela.currentRow() >= 0
        self.btn_editar.setEnabled(selecionado)
        self.btn_excluir.setEnabled(selecionado)

    def criar_prestador(self):
        dialog = DialogPrestador()
        if dialog.exec_():
            nome, doc, servico, valor, status = dialog.get_data()
            conn = self.conexao()
            cursor = conn.cursor()
            cursor.execute(
                "INSERT INTO prestadores (nome, cpf_cnpj, tipo_servico, valor_mensal, status) VALUES (?, ?, ?, ?, ?)",
                (nome, doc, servico, valor, status)
            )
            conn.commit()
            conn.close()
            self.carregar_dados()

    def editar_prestador(self):
        row = self.tabela.currentRow()
        if row < 0:
            return
        nome = self.tabela.item(row, 0).text()
        conn = self.conexao()
        cursor = conn.cursor()
        cursor.execute("SELECT * FROM prestadores WHERE nome = ?", (nome,))
        prestador = cursor.fetchone()
        conn.close()

        dialog = DialogPrestador(prestador)
        if dialog.exec_():
            novo_nome, doc, servico, valor, status = dialog.get_data()
            conn = self.conexao()
            cursor = conn.cursor()
            cursor.execute("""
                UPDATE prestadores 
                SET nome = ?, cpf_cnpj = ?, tipo_servico = ?, valor_mensal = ?, status = ? 
                WHERE id = ?
            """, (novo_nome, doc, servico, valor, status, prestador[0]))
            conn.commit()
            conn.close()
            self.carregar_dados()

    def excluir_prestador(self):
        row = self.tabela.currentRow()
        if row < 0:
            return
        nome = self.tabela.item(row, 0).text()
        confirm = QMessageBox.question(self, "Confirmar Exclusão",
            f"Tem certeza que deseja excluir o prestador '{nome}'?", QMessageBox.Yes | QMessageBox.No)
        if confirm == QMessageBox.Yes:
            conn = self.conexao()
            cursor = conn.cursor()
            cursor.execute("DELETE FROM prestadores WHERE nome = ?", (nome,))
            conn.commit()
            conn.close()
            self.carregar_dados()

# ----------------------------
# Aba 2 - Pagamentos Mensais
# ----------------------------


class DialogPagamento(QDialog):
    def __init__(self, prestadores_ativos, dados_pagamento=None, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Lançar Pagamento")
        self.setMinimumWidth(500)

        layout_principal = QVBoxLayout()
        form_layout = QFormLayout()

        self.combo_prestador = QComboBox()
        for p in prestadores_ativos:
            self.combo_prestador.addItem(p[1], p[0])  # nome, id

        # Substituir QLineEdit por dois ComboBox
        meses = ["01", "02", "03", "04", "05", "06", "07", "08", "09", "10", "11", "12"]
        anos = [str(ano) for ano in range(2025, QDate.currentDate().year() + 2)]

        self.combo_mes = QComboBox()
        self.combo_mes.addItems(meses)
        self.combo_ano = QComboBox()
        self.combo_ano.addItems(anos)

        competencia_layout = QHBoxLayout()
        competencia_layout.addWidget(self.combo_mes)
        competencia_layout.addWidget(QLabel("/"))
        competencia_layout.addWidget(self.combo_ano)

        self.input_valor_pago = QLineEdit()
        validator_pgto = QDoubleValidator(0.00, 999999.99, 2)
        validator_pgto.setNotation(QDoubleValidator.StandardNotation)
        self.input_valor_pago.setValidator(validator_pgto)

        self.data_pagamento = QDateEdit()
        self.data_pagamento.setCalendarPopup(True)
        self.data_pagamento.setDate(QDate.currentDate())
        self.input_observacao = QLineEdit()

        form_layout.addRow("Prestador:", self.combo_prestador)
        form_layout.addRow("Competência:", competencia_layout)
        form_layout.addRow("Valor Pago (R$):", self.input_valor_pago)
        form_layout.addRow("Data Pagamento:", self.data_pagamento)
        form_layout.addRow("Observação:", self.input_observacao)

        layout_principal.addLayout(form_layout)

        # Botão centralizado
        btn_salvar = QPushButton("Salvar")
        btn_salvar.setFixedWidth(120)
        btn_salvar.clicked.connect(self.accept)
        btn_layout = QHBoxLayout()
        btn_layout.addStretch()
        btn_layout.addWidget(btn_salvar)
        btn_layout.addStretch()
        layout_principal.addLayout(btn_layout)

        self.setLayout(layout_principal)

        # Preenche os dados caso seja edição
        if dados_pagamento:
            self.combo_prestador.setCurrentIndex(
                self.combo_prestador.findData(dados_pagamento["id_prestador"])
            )
            mes, ano = dados_pagamento["competencia"].split("/")
            self.combo_mes.setCurrentText(mes)
            self.combo_ano.setCurrentText(ano)

            # garante formatação numérica simples
            self.input_valor_pago.setText(f'{float(dados_pagamento["valor_pago"]):.2f}')

            # QDateEdit precisa de QDate
            dt = datetime.strptime(dados_pagamento["data_pagamento"], "%Y-%m-%d")
            self.data_pagamento.setDate(QDate(dt.year, dt.month, dt.day))

            self.input_observacao.setText(dados_pagamento.get("observacao", ""))


    def get_data(self):
        valor_texto = self.input_valor_pago.text().replace(",", ".").strip()

        if not valor_texto:
            QMessageBox.warning(self, "Campo obrigatório", "Informe o valor pago.")
            return None

        try:
            valor_digitado = float(valor_texto)
        except ValueError:
            QMessageBox.warning(self, "Erro", "Valor inválido informado.")
            return None

        if valor_digitado < 0:
            QMessageBox.warning(self, "Erro", "O valor pago não pode ser negativo.")
            return None

        prestador_id = self.combo_prestador.currentData()
        conn = sqlite3.connect(CAMINHO_BANCO)
        cursor = conn.cursor()
        cursor.execute("SELECT valor_mensal FROM prestadores WHERE id = ?", (prestador_id,))
        resultado = cursor.fetchone()
        conn.close()

        if resultado:
            valor_mensal = resultado[0]
            if valor_digitado > valor_mensal:
                QMessageBox.warning(
                    self,
                    "Valor excedente",
                    f"O valor pago (R$ {valor_digitado:.2f}) excede o valor mensal do prestador (R$ {valor_mensal:.2f})."
                )
                return None
        else:
            QMessageBox.warning(self, "Erro", "Não foi possível localizar o prestador.")
            return None

        # 🧠 Determinar o status automaticamente
        if valor_digitado == 0:
            status = "Não pago"
        elif valor_digitado < valor_mensal:
            status = "Parcial"
        else:
            status = "Pago"

        return {
            "id_prestador": prestador_id,
            "competencia": f"{self.combo_mes.currentText()}/{self.combo_ano.currentText()}",
            "valor_pago": valor_digitado,
            "data_pagamento": self.data_pagamento.date().toString("yyyy-MM-dd"),
            "observacao": self.input_observacao.text().strip(),
            "status_pagamento": status
        }




class AbaPagamentosPrestadores(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Pagamentos de Prestadores")
        self.layout = QVBoxLayout(self)

        filtro_layout = QHBoxLayout()
        filtro_layout.addWidget(QLabel("Competência:"))
        self.combo_competencia = QComboBox()
        self.combo_competencia.currentIndexChanged.connect(self.carregar_pagamentos)
        filtro_layout.addWidget(QLabel(" | Filtrar por nome:"))
        self.input_filtro_nome = QLineEdit()
        self.input_filtro_nome.setPlaceholderText("🔍 Digite o nome do prestador...")
        self.input_filtro_nome.textChanged.connect(self.carregar_pagamentos)
        filtro_layout.addWidget(self.input_filtro_nome)

        filtro_layout.addWidget(self.combo_competencia)

        self.btn_lancar = QPushButton("Lançar Pagamento")
        self.btn_editar = QPushButton("Editar Pagamento")
        self.btn_editar.setEnabled(False)
        self.btn_excluir = QPushButton("Excluir Pagamento")
        self.btn_excluir.setEnabled(False)

        
        filtro_layout.addWidget(self.btn_lancar)
        filtro_layout.addWidget(self.btn_editar)
        filtro_layout.addWidget(self.btn_excluir)
        self.layout.addLayout(filtro_layout)

        self.tabela = QTableWidget()
        self.tabela.setColumnCount(7)
        self.tabela.setHorizontalHeaderLabels(["Prestador", "Valor Pago", "Valor em Aberto", "Status", "Data", "Observação", "Competência"])
        self.tabela.setSelectionBehavior(QTableWidget.SelectRows)
        self.tabela.itemSelectionChanged.connect(self.habilitar_editar)
        self.layout.addWidget(self.tabela)

        # 🛠️ Ajuste visual da tabela
        self.tabela.horizontalHeader().setStretchLastSection(True)
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)

        self.btn_lancar.clicked.connect(self.lancar_pagamento)
        self.btn_editar.clicked.connect(self.editar_pagamento)
        self.btn_excluir.clicked.connect(self.excluir_pagamento)


        self.label_rodape = QLabel("Valor Pago: R$ 0,00 | Valor em Aberto: R$ 0,00")
        self.layout.addWidget(self.label_rodape)

        # Ajuste visual da tabela
        self.tabela.horizontalHeader().setStretchLastSection(True)
        self.tabela.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)


        self.carregar_competencias()


    def conexao(self):
        return sqlite3.connect(CAMINHO_BANCO)

    def carregar_competencias(self):
        conn = self.conexao()
        c = conn.cursor()
        c.execute("SELECT DISTINCT competencia FROM pagamentos_prestadores ORDER BY competencia DESC")
        competencias = [row[0] for row in c.fetchall()]
        conn.close()
        self.combo_competencia.clear()
        self.combo_competencia.addItems(competencias if competencias else ["07/2025"])



    def carregar_pagamentos(self):
        competencia = self.combo_competencia.currentText()
        conn = self.conexao()
        c = conn.cursor()
        filtro_nome = self.input_filtro_nome.text().strip().lower()

        query = """
            SELECT p.nome, pg.valor_pago, pg.status_pagamento, pg.data_pagamento, pg.observacao, pg.competencia
            FROM pagamentos_prestadores pg
            JOIN prestadores p ON pg.id_prestador = p.id
            WHERE pg.competencia = ?
        """
        params = [competencia]

        if filtro_nome:
            query += " AND lower(p.nome) LIKE ?"
            params.append(f"%{filtro_nome}%")

        query += " ORDER BY p.nome"
        c.execute(query, params)

        dados = c.fetchall()

        self.tabela.setRowCount(0)
        total_pago = 0.0
        total_em_aberto = 0.0

        for i, row in enumerate(dados):
            prestador_nome, valor_pago, status, data, obs, comp = row

            # Buscar valor mensal
            c.execute("SELECT valor_mensal FROM prestadores WHERE nome = ?", (prestador_nome,))
            resultado = c.fetchone()
            valor_mensal = resultado[0] if resultado else 0.0

            # Calcular valor em aberto
            valor_em_aberto = max(valor_mensal - valor_pago, 0)

            # Soma para rodapé
            total_pago += valor_pago
            total_em_aberto += valor_em_aberto

            # Monta linha da tabela
            self.tabela.insertRow(i)

            colunas = [
                prestador_nome,
                f"{valor_pago:.2f}",
                f"{valor_em_aberto:.2f}",
                "✅ Pago" if status == "Pago" else "🟡 Parcial" if status == "Parcial" else "❌ Não pago",
                data,
                obs,
                comp
            ]

            for j, valor in enumerate(colunas):
                item = QTableWidgetItem(str(valor))
                item.setTextAlignment(Qt.AlignCenter)
                self.tabela.setItem(i, j, item)

        conn.close()

        # Atualizar rodapé
        valor_pago_str = f"R$ {total_pago:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        valor_aberto_str = f"R$ {total_em_aberto:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")
        self.label_rodape.setText(
            f"<b>Valor Pago:</b> {valor_pago_str} &nbsp;&nbsp;|&nbsp;&nbsp; <b>Valor em Aberto:</b> {valor_aberto_str}"
        )



    def obter_prestadores_ativos(self):
        conn = self.conexao()
        c = conn.cursor()
        c.execute("SELECT id, nome FROM prestadores WHERE status = 'Ativo'")
        return c.fetchall()

    def habilitar_editar(self):
        selecionado = self.tabela.currentRow() >= 0
        self.btn_editar.setEnabled(selecionado)
        self.btn_excluir.setEnabled(selecionado)


    def lancar_pagamento(self):
        prestadores = self.obter_prestadores_ativos()
        if not prestadores:
            QMessageBox.warning(self, "Aviso", "Não há prestadores ativos para lançar pagamento.")
            return

        dialog = DialogPagamento(prestadores)
        if dialog.exec_():
            dados = dialog.get_data()
            if not dados:
                # o próprio diálogo já mostrou o motivo
                return

            try:
                conn = self.conexao()
                c = conn.cursor()
                c.execute("""
                    INSERT INTO pagamentos_prestadores 
                    (id_prestador, competencia, valor_pago, data_pagamento, observacao, status_pagamento)
                    VALUES (?, ?, ?, ?, ?, ?)
                """, (
                    dados["id_prestador"], dados["competencia"], dados["valor_pago"],
                    dados["data_pagamento"], dados["observacao"], dados["status_pagamento"]
                ))
                conn.commit()
            except sqlite3.Error as e:
                QMessageBox.critical(self, "Erro", f"Falha ao salvar pagamento:\n{e}")
            finally:
                conn.close()

            self.carregar_competencias()
            self.carregar_pagamentos()


    def editar_pagamento(self):
        row = self.tabela.currentRow()
        if row < 0:
            return

        nome = self.tabela.item(row, 0).text()
        competencia = self.tabela.item(row, 6).text()  # ← coluna "Competência" agora é a 6ª posição (índice 6)

        conn = self.conexao()
        c = conn.cursor()
        c.execute("""
            SELECT pg.id, pg.id_prestador, pg.competencia, pg.valor_pago, pg.data_pagamento, pg.observacao, pg.status_pagamento
            FROM pagamentos_prestadores pg
            JOIN prestadores p ON pg.id_prestador = p.id
            WHERE pg.competencia = ? AND p.nome = ?
        """, (competencia, nome))
        row_db = c.fetchone()
        conn.close()

        if not row_db:
            QMessageBox.warning(self, "Erro", "Não foi possível localizar o registro para edição.")
            return

        dados = {
            "id": row_db[0],
            "id_prestador": row_db[1],
            "competencia": row_db[2],
            "valor_pago": row_db[3],
            "data_pagamento": row_db[4],
            "observacao": row_db[5],
            "status_pagamento": row_db[6]
        }

        dialog = DialogPagamento(self.obter_prestadores_ativos(), dados)
        if dialog.exec_():
            novos_dados = dialog.get_data()
            if not novos_dados:
                return

            try:
                conn = self.conexao()
                c = conn.cursor()
                c.execute("""
                    UPDATE pagamentos_prestadores
                    SET id_prestador = ?, competencia = ?, valor_pago = ?, data_pagamento = ?, observacao = ?, status_pagamento = ?
                    WHERE id = ?
                """, (
                    novos_dados["id_prestador"], novos_dados["competencia"], novos_dados["valor_pago"],
                    novos_dados["data_pagamento"], novos_dados["observacao"], novos_dados["status_pagamento"],
                    dados["id"]
                ))
                conn.commit()
            except sqlite3.Error as e:
                QMessageBox.critical(self, "Erro", f"Falha ao atualizar pagamento:\n{e}")
            finally:
                conn.close()

            self.carregar_pagamentos()


    def excluir_pagamento(self):
        row = self.tabela.currentRow()
        if row < 0:
            return

        nome = self.tabela.item(row, 0).text()
        competencia = self.tabela.item(row, 6).text()

        confirm = QMessageBox.question(
            self,
            "Confirmar Exclusão",
            f"Deseja excluir o pagamento do prestador '{nome}' referente à competência {competencia}?",
            QMessageBox.Yes | QMessageBox.No
        )
        if confirm != QMessageBox.Yes:
            return

        conn = self.conexao()
        c = conn.cursor()
        c.execute("""
            SELECT pg.id
            FROM pagamentos_prestadores pg
            JOIN prestadores p ON pg.id_prestador = p.id
            WHERE pg.competencia = ? AND p.nome = ?
        """, (competencia, nome))
        resultado = c.fetchone()

        if resultado:
            pagamento_id = resultado[0]
            c.execute("DELETE FROM pagamentos_prestadores WHERE id = ?", (pagamento_id,))
            conn.commit()

        conn.close()
        self.carregar_pagamentos()



# ----------------------------
# Integração Final: Módulo Completo
# ----------------------------

class ModuloPrestadores(QWidget):
    def __init__(self):
        super().__init__()
        layout = QVBoxLayout(self)
        abas = QTabWidget()
        abas.addTab(AbaPrestadores(), "Cadastro de Prestadores")
        abas.addTab(AbaPagamentosPrestadores(), "Pagamentos Mensais")
        layout.addWidget(abas)
        self.setLayout(layout)