from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QPushButton, QTableWidget,
    QTableWidgetItem, QDialog, QFormLayout, QLineEdit, QComboBox,
    QMessageBox, QSpinBox, QHeaderView, QLabel, QFrame, QAbstractItemView
)
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QDoubleValidator
import sqlite3, os

CAMINHO_BANCO = os.path.join("db", "sistema_financeiro.db")

def _coluna_existe(cur, tabela, coluna) -> bool:
    cur.execute(f"PRAGMA table_info({tabela})")
    return any(r[1] == coluna for r in cur.fetchall())

def ensure_schema_impostos():
    """Garante as colunas de vigência/periodicidade e a tabela/índices usados no módulo."""
    dbpath = os.path.abspath(CAMINHO_BANCO)
    with sqlite3.connect(CAMINHO_BANCO) as con:
        cur = con.cursor()

        # Vigências
        if not _coluna_existe(cur, "cadastro_impostos", "vigencia_inicio"):
            cur.execute("ALTER TABLE cadastro_impostos ADD COLUMN vigencia_inicio TEXT")  # 'YYYY-MM'
        if not _coluna_existe(cur, "cadastro_impostos", "vigencia_fim"):
            cur.execute("ALTER TABLE cadastro_impostos ADD COLUMN vigencia_fim TEXT")     # 'YYYY-MM'

        # 🔹 Periodicidade
        if not _coluna_existe(cur, "cadastro_impostos", "periodicidade"):
            cur.execute("ALTER TABLE cadastro_impostos ADD COLUMN periodicidade TEXT DEFAULT 'Mensal'")
        if not _coluna_existe(cur, "cadastro_impostos", "dia_semana"):
            cur.execute("ALTER TABLE cadastro_impostos ADD COLUMN dia_semana INTEGER")    # 0=Seg ... 6=Dom
        if not _coluna_existe(cur, "cadastro_impostos", "mes_vencimento"):
            cur.execute("ALTER TABLE cadastro_impostos ADD COLUMN mes_vencimento INTEGER")  # 1..12

        # Tabela de competências
        cur.execute("""
            CREATE TABLE IF NOT EXISTS impostos_competencia (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                imposto_id INTEGER,
                competencia TEXT,   -- 'MM/AAAA' (usada pra filtrar a tela)
                valor REAL,
                vencimento TEXT,    -- 'YYYY-MM-DD' (usada pra unicidade e status)
                pago INTEGER DEFAULT 0
            )
        """)

        # 🔄 Ajuste do índice único:
        # Antes: UNIQUE(imposto_id, competencia) -> impede semanal (várias no mês)
        # Agora: UNIQUE(imposto_id, vencimento)
        cur.execute("DROP INDEX IF EXISTS ux_imp_comp")
        cur.execute("""
            CREATE UNIQUE INDEX IF NOT EXISTS ux_imp_venc
            ON impostos_competencia(imposto_id, vencimento)
        """)
        # Índice auxiliar pra filtrar rápido por competência
        cur.execute("""
            CREATE INDEX IF NOT EXISTS idx_imp_competencia
            ON impostos_competencia(competencia)
        """)

        con.commit()
    print("Schema verificado/ajustado em:", dbpath)



def fmt_money(v):
    s = f"{float(v):,.2f}"
    s = s.replace(",", "X").replace(".", ",").replace("X", ".")
    return f"R$ {s}"

# ---------------- Dialogo de Cadastro/Edição ----------------
class DialogCadastroImposto(QDialog):
    def __init__(self, imposto=None, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Cadastro de Imposto")
        self.setFixedSize(380, 360)

        layout = QFormLayout(self)

        # --- Campos base ---
        self.input_nome = QLineEdit()

        self.combo_origem = QComboBox()
        self.combo_origem.addItems(["Nacional", "Estadual", "Municipal"])

        # Periodicidade (apenas Mensal e Anual)
        self.combo_periodicidade = QComboBox()
        self.combo_periodicidade.addItems(["Mensal", "Anual"])

        # Valor padrão
        self.input_valor = QLineEdit()
        self.input_valor.setPlaceholderText("0,00")
        self.input_valor.setValidator(QDoubleValidator(0.0, 1e12, 2))

        # Dia do vencimento (Mensal/Anual)
        self.input_vencimento = QSpinBox()
        self.input_vencimento.setRange(1, 31)
        self.input_vencimento.setValue(20)

        # Mês do vencimento (Anual)
        self.combo_mes_anual = QComboBox()
        self.combo_mes_anual.addItems([f"{m:02d}" for m in range(1, 13)])

        # Ativo
        self.combo_ativo = QComboBox()
        self.combo_ativo.addItems(["Ativo", "Inativo"])

        # Vigência (início obrigatório, fim opcional)
        self.combo_ini_mes = QComboBox(); self.combo_ini_mes.addItems([f"{m:02d}" for m in range(1, 13)])
        self.combo_ini_ano = QComboBox(); self.combo_ini_ano.addItems([str(a) for a in range(2025, 2032)])
        self.combo_fim_mes = QComboBox(); self.combo_fim_mes.addItems(["--"] + [f"{m:02d}" for m in range(1, 13)])
        self.combo_fim_ano = QComboBox(); self.combo_fim_ano.addItems(["----"] + [str(a) for a in range(2025, 2032)])

        # --- Montagem do formulário (uma vez só) ---
        layout.addRow("Nome:", self.input_nome)
        layout.addRow("Origem:", self.combo_origem)
        layout.addRow("Periodicidade:", self.combo_periodicidade)

        # Linhas dependentes da periodicidade
        self.row_mensal_anual = self._h(QLabel("Dia venc."), self.input_vencimento)  # sempre visível
        self.row_mes_anual    = self._h(QLabel("Mês venc."), self.combo_mes_anual)   # só no Anual
        layout.addRow(self.row_mensal_anual)
        layout.addRow(self.row_mes_anual)

        layout.addRow("Valor Padrão:", self.input_valor)
        layout.addRow("Status:", self.combo_ativo)
        layout.addRow("Aparecer a partir de:", self._h(self.combo_ini_mes, QLabel("/"), self.combo_ini_ano))
        layout.addRow("Parar em (opcional):", self._h(self.combo_fim_mes, QLabel("/"), self.combo_fim_ano))

        btn_salvar = QPushButton("Salvar")
        btn_salvar.clicked.connect(self.accept)
        layout.addRow(btn_salvar)

        # Mostrar/ocultar campos conforme periodicidade
        def _update_fields_by_periodicidade():
            per = self.combo_periodicidade.currentText()
            self.row_mensal_anual.setVisible(True)        # dia sempre usado
            self.row_mes_anual.setVisible(per == "Anual") # mês só no Anual
        self.combo_periodicidade.currentIndexChanged.connect(_update_fields_by_periodicidade)
        _update_fields_by_periodicidade()

        # Preenchimento em modo edição (usa colunas novas)
        if imposto:
            # (id, nome, origem, valor, dia_venc, ativo, vig_ini, vig_fim, periodicidade, dia_semana, mes_venc)
            self.input_nome.setText(imposto[1])
            self.combo_origem.setCurrentText(imposto[2])
            self.input_valor.setText(str(imposto[3]).replace(".", ","))
            self.input_vencimento.setValue(int(imposto[4]) if imposto[4] else 20)
            self.combo_ativo.setCurrentIndex(0 if int(imposto[5]) == 1 else 1)

            vig_ini = (imposto[6] or "").split("-")
            if len(vig_ini) == 2:
                self.combo_ini_ano.setCurrentText(vig_ini[0])
                self.combo_ini_mes.setCurrentText(vig_ini[1])

            vig_fim = (imposto[7] or "")
            if vig_fim:
                parts = vig_fim.split("-")
                if len(parts) == 2:
                    self.combo_fim_ano.setCurrentText(parts[0])
                    self.combo_fim_mes.setCurrentText(parts[1])

            self.combo_periodicidade.setCurrentText(imposto[8] or "Mensal")
            if imposto[10] is not None:
                self.combo_mes_anual.setCurrentText(f"{int(imposto[10]):02d}")
            _update_fields_by_periodicidade()

    def _h(self, *widgets):
        w = QWidget(); lay = QHBoxLayout(w); lay.setContentsMargins(0,0,0,0); lay.setSpacing(6)
        for wd in widgets: lay.addWidget(wd)
        lay.addStretch(1)
        return w

    def get_dados(self):
        # monta tupla com 10 campos (mantém compatibilidade do schema)
        valor_txt = self.input_valor.text().strip().replace(".", "").replace(",", ".")
        valor = float(valor_txt) if valor_txt else 0.0

        nome = self.input_nome.text().strip()
        origem = self.combo_origem.currentText().strip()
        ativo = 1 if self.combo_ativo.currentText() == "Ativo" else 0

        per = self.combo_periodicidade.currentText().strip()  # 'Mensal' | 'Anual'
        dia_venc = int(self.input_vencimento.value())
        mes_anual = int(self.combo_mes_anual.currentText()) if per == "Anual" else None

        vig_ini = f"{self.combo_ini_ano.currentText()}-{self.combo_ini_mes.currentText()}"
        vig_fim = None
        fim_ano = self.combo_fim_ano.currentText().strip()
        fim_mes = self.combo_fim_mes.currentText().strip()
        if fim_ano and fim_mes and fim_ano != "----" and fim_mes != "--":
            vig_fim = f"{fim_ano}-{fim_mes}"

        dia_semana = None  # mantemos coluna no banco, mas não usamos
        return (nome, origem, valor, dia_venc, ativo, vig_ini, vig_fim, per, dia_semana, mes_anual)




# ---------------- Tela de Lista/CRUD ----------------
class ModuloCadastroImpostos(QWidget):
    def __init__(self):
        super().__init__()
        ensure_schema_impostos()

        # Layout base padronizado
        self.layout = QVBoxLayout(self)
        self.layout.setSpacing(20)
        self.layout.setContentsMargins(20, 20, 20, 20)

        estilo_botao = """
            QPushButton {
                background-color: #3498db;
                color: white;
                padding: 6px 12px;
                font-weight: bold;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #2980b9;
            }
        """

        # Botões (padronizados)
        botoes_layout = QHBoxLayout()
        self.btn_novo    = QPushButton("➕ Novo Imposto")
        self.btn_editar  = QPushButton("✏️ Editar Imposto")
        self.btn_excluir = QPushButton("🗑️ Excluir Imposto")
        for b in (self.btn_novo, self.btn_editar, self.btn_excluir):
            b.setStyleSheet(estilo_botao)

        self.btn_novo.clicked.connect(self.novo_imposto)
        self.btn_editar.clicked.connect(self.editar_imposto)
        self.btn_excluir.clicked.connect(self.excluir_imposto)

        botoes_layout.addWidget(self.btn_novo)
        botoes_layout.addWidget(self.btn_editar)
        botoes_layout.addWidget(self.btn_excluir)
        botoes_layout.addStretch(1)

        # Campo de busca
        filtro_layout = QHBoxLayout()
        self.campo_busca = QLineEdit()
        self.campo_busca.setPlaceholderText("🔍 Buscar por nome/origem/regra de vencimento...")
        self.campo_busca.textChanged.connect(self.filtrar_impostos)
        filtro_layout.addWidget(self.campo_busca)

        # Separador
        linha_div = QFrame()
        linha_div.setFrameShape(QFrame.HLine)
        linha_div.setFrameShadow(QFrame.Sunken)

        # Tabela
        self.tabela = QTableWidget()
        self.tabela.setColumnCount(5)
        self.tabela.setHorizontalHeaderLabels(["Nome", "Origem", "Valor Padrão", "Regra de Venc.", "Status"])
        self.tabela.verticalHeader().setVisible(False)
        self.tabela.setAlternatingRowColors(True)
        self.tabela.setSortingEnabled(True)
        
        # ⬇️ igual aos outros módulos        
        header = self.tabela.horizontalHeader()
        header.setSectionResizeMode(QHeaderView.Stretch)
        header.setDefaultAlignment(Qt.AlignCenter)
        
        header = self.tabela.horizontalHeader()
        header.setSectionResizeMode(QHeaderView.Stretch)
        header.setDefaultAlignment(Qt.AlignCenter)

        # Montagem final
        self.layout.addLayout(botoes_layout)
        self.layout.addLayout(filtro_layout)
        self.layout.addWidget(linha_div)
        self.layout.addWidget(self.tabela)

        # Carregar dados
        self.carregar_dados()
        
        
    def filtrar_impostos(self):
        texto = (self.campo_busca.text() or "").strip().lower()
        termos = [t for t in texto.split() if t]  # permite múltiplas palavras
        for row in range(self.tabela.rowCount()):
            blob = " ".join(
                (self.tabela.item(row, c).text().lower() if self.tabela.item(row, c) else "")
                for c in range(self.tabela.columnCount())
            )
            visivel = all(t in blob for t in termos)
            self.tabela.setRowHidden(row, not visivel)



    def carregar_dados(self):
        self.tabela.setRowCount(0)
        with sqlite3.connect(CAMINHO_BANCO) as con:
            cur = con.cursor()
            cur.execute("""
                SELECT id, nome, origem, valor, dia_vencimento, ativo,
                    vigencia_inicio, vigencia_fim, periodicidade, mes_vencimento
                FROM cadastro_impostos
                ORDER BY nome
            """)
            rows = cur.fetchall()

        def _regra_venc(per, dia_venc, mes_anual):
            if (per or "Mensal") == "Anual":
                return f"{(dia_venc or 20):02d}/{(mes_anual or 1):02d}"
            return f"dia {dia_venc or 20}"

        for row_idx, (iid, nome, origem, valor, dia_venc, ativo, vig_ini, vig_fim, per, mes_anual) in enumerate(rows):
            self.tabela.insertRow(row_idx)

            item_nome = QTableWidgetItem(nome)
            item_nome.setData(Qt.UserRole, iid)
            item_nome.setTextAlignment(Qt.AlignCenter)
            item_nome.setToolTip(f"Vigência: {vig_ini or 'sempre'} até {vig_fim or '—'}")
            self.tabela.setItem(row_idx, 0, item_nome)

            item_origem = QTableWidgetItem(origem); item_origem.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 1, item_origem)

            item_valor = QTableWidgetItem(fmt_money(valor)); item_valor.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 2, item_valor)

            item_venc = QTableWidgetItem(_regra_venc(per, dia_venc, mes_anual)); item_venc.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 3, item_venc)

            item_status = QTableWidgetItem("Ativo" if int(ativo) == 1 else "Inativo"); item_status.setTextAlignment(Qt.AlignCenter)
            self.tabela.setItem(row_idx, 4, item_status)



    def _get_selected_imposto_id(self):
        linha = self.tabela.currentRow()
        if linha < 0:
            return None
        itm = self.tabela.item(linha, 0)
        return None if not itm else itm.data(Qt.UserRole)

    def novo_imposto(self):
        dialog = DialogCadastroImposto()
        if dialog.exec_():
            (nome, origem, valor, dia_vencimento, ativo, vig_ini, vig_fim,
            periodicidade, dia_semana, mes_vencimento) = dialog.get_dados()

            if not nome:
                QMessageBox.warning(self, "Atenção", "Informe o nome do imposto.")
                return

            with sqlite3.connect(CAMINHO_BANCO) as con:
                cur = con.cursor()
                cur.execute("""
                    INSERT INTO cadastro_impostos
                    (nome, origem, valor, dia_vencimento, ativo, vigencia_inicio, vigencia_fim,
                    periodicidade, dia_semana, mes_vencimento)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                """, (nome, origem, valor, dia_vencimento, ativo, vig_ini, vig_fim,
                    periodicidade, dia_semana, mes_vencimento))
                con.commit()
            self.carregar_dados()



    def editar_imposto(self):
        imposto_id = self._get_selected_imposto_id()
        if imposto_id is None:
            QMessageBox.warning(self, "Atenção", "Selecione um imposto para editar.")
            return

        with sqlite3.connect(CAMINHO_BANCO) as con:
            cur = con.cursor()
            cur.execute("""
                SELECT id, nome, origem, valor, dia_vencimento, ativo, vigencia_inicio, vigencia_fim,
                    periodicidade, dia_semana, mes_vencimento
                FROM cadastro_impostos
                WHERE id=?
            """, (imposto_id,))
            imposto = cur.fetchone()

        dialog = DialogCadastroImposto(imposto)
        if dialog.exec_():
            (nome, origem, valor, dia_vencimento, ativo, vig_ini, vig_fim,
            periodicidade, dia_semana, mes_vencimento) = dialog.get_dados()

            if not nome:
                QMessageBox.warning(self, "Atenção", "Informe o nome do imposto.")
                return

            with sqlite3.connect(CAMINHO_BANCO) as con:
                cur = con.cursor()
                cur.execute("""
                    UPDATE cadastro_impostos
                    SET nome=?, origem=?, valor=?, dia_vencimento=?, ativo=?,
                        vigencia_inicio=?, vigencia_fim=?,
                        periodicidade=?, dia_semana=?, mes_vencimento=?
                    WHERE id=?
                """, (nome, origem, valor, dia_vencimento, ativo,
                    vig_ini, vig_fim, periodicidade, dia_semana, mes_vencimento, imposto_id))
                con.commit()
            self.carregar_dados()


    def excluir_imposto(self):
        imposto_id = self._get_selected_imposto_id()
        if imposto_id is None:
            QMessageBox.warning(self, "Atenção", "Selecione um imposto para excluir.")
            return

        nome = self.tabela.item(self.tabela.currentRow(), 0).text()
        confirm = QMessageBox.question(
            self, "Confirmar", f"Deseja excluir '{nome}'?",
            QMessageBox.Yes | QMessageBox.No
        )
        if confirm == QMessageBox.Yes:
            con = sqlite3.connect(CAMINHO_BANCO)
            cur = con.cursor()
            cur.execute("DELETE FROM cadastro_impostos WHERE id=?", (imposto_id,))
            con.commit()
            con.close()
            self.carregar_dados()
